///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file LookAtController.h 
 * \brief Contains the definition of the Core::LookAtController class. 
 */

#ifndef __OVITO_LOOKAT_CONTROLLER_H
#define __OVITO_LOOKAT_CONTROLLER_H

#include <core/Core.h>
#include "Controller.h"
#include <core/scene/SceneNode.h>

namespace Core {

/**
 * \brief Rotation controller that lets an object always "look" at another scene object.
 * 
 * This RotationController computes a rotation matrix for a SceneNode such
 * that it always faces into the direction of another SceneNode.
 * 
 * \author Alexander Stukowski
 * \sa SceneNode::bindToTarget()
 * \sa AffineTransformation::lookAt()
 */
class CORE_DLLEXPORT LookAtController : public RotationController
{
public:
	
	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file. 
	///                  This parameter is only used by the object serialization system.
    LookAtController(bool isLoading = false);

	/// Queries the controller for its absolute value at a certain time.
	virtual void getValue(TimeTicks time, Rotation& result, TimeInterval& validityInterval);

	/// Sets the controller's value at the specified time.
	virtual void setValue(TimeTicks time, const Rotation& newValue, bool isAbsoluteValue);

	/// Let the controller add its value at a certain time to the input value.
	virtual void applyValue(TimeTicks time, AffineTransformation& result, TimeInterval& validityInterval);

	/// Computes the largest time interval containing the given time during which the
	/// controller's value is constant.
	virtual void validityInterval(TimeTicks time, TimeInterval& validityInterval);

	/// This asks the controller to adjust its value after a scene node has got a new
	/// parent node.
	///		oldParentTM - The transformation of the old parent node
	///		newParentTM - The transformation of the new parent node
	///		contextNode - The node to which this cotnroller is assigned to
	virtual void changeParent(TimeTicks time, const AffineTransformation& oldParentTM, const AffineTransformation& newParentTM, SceneNode* contextNode) {}

	/// \brief Returns the target node to look at.
	/// \return The target this rotation controller is looking at or \c NULL if it
	///         has not been set yet.
	/// \sa setTargetNode()
	/// \sa SceneNode::targetNode()
	SceneNode* targetNode() const { return _targetNode; }

	/// \brief Sets the target node to look at.
	/// \param target The new target this rotation controller should use for its
	///               calculation of the rotation matrix.
	/// \undoable
	void setTargetNode(SceneNode* target) { _targetNode = target; }

	/// \brief Sets the target node to look at.
	/// \param target The new target this rotation controller should use for its
	///               calculation of the rotation matrix.
	/// \note This is the same method as above but takes a smart pointer instead of a raw pointer.
	/// \undoable
	/// \sa SceneNode::bindToTarget()
	/// \sa targetNode()
	void setTargetNode(const SceneNode::SmartPtr& target) { setTargetNode(target.get()); }
	
	/// \brief Returs the sub-controller that controls the rolling parameter.
	/// \return The sub-controller for the rolling angle.
	/// \sa setRollController()
	FloatController* rollController() const { return _rollCtrl; }

	/// \brief Sets the sub-controller that controls the rolling parameter.
	/// \param ctrl The new roll angle controller. 
	/// \undoable
	void setRollController(FloatController* ctrl) { _rollCtrl = ctrl; }

	/// \brief Sets the sub-controller that controls the rolling parameter.
	/// \param ctrl The new roll angle controller. 
	/// \note This is the same method as above but takes a smart pointer instead of a raw pointer.
	/// \undoable
	void setRollController(const FloatController::SmartPtr& ctrl) { setRollController(ctrl.get()); }
	
private:

	/// The sub-controller for rolling.
	ReferenceField<FloatController> _rollCtrl;

	/// The target scene node to look at.
	ReferenceField<SceneNode> _targetNode;

	/// Stores the position of the source node.
	Vector3 sourcePos;

	/// Stores the validity interval of the saved source position.
	TimeInterval sourcePosValidity;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LookAtController)
	DECLARE_REFERENCE_FIELD(_rollCtrl)
	DECLARE_REFERENCE_FIELD(_targetNode)
};

};

#endif // __OVITO_LOOKAT_CONTROLLER_H
