# go-fuzz: randomized testing for Go

Go-fuzz is a coverage-guided [fuzzing solution](http://en.wikipedia.org/wiki/Fuzz_testing) for testing of Go packages.
Fuzzing is mainly applicable to packages that parse complex inputs (both text
and binary), and is especially useful for hardening of systems that parse inputs
from potentially malicious users (e.g. anything accepted over a network).

**Note:** go-fuzz has recently added preliminary support for fuzzing [Go Modules](https://github.com/golang/go/wiki/Modules).  See the [section below](https://github.com/dvyukov/go-fuzz/blob/master/README.md#modules-support) for more details. 
If you encounter a problem with modules, please file an issue with details. A workaround might be to disable modules via `export GO111MODULE=off`.

## Usage

First, you need to write a test function of the form:
```go
func Fuzz(data []byte) int
```
Data is a random input generated by go-fuzz, note that in most cases it is
invalid. The function must return 1 if the fuzzer should increase priority
of the given input during subsequent fuzzing (for example, the input is
lexically correct and was parsed successfully); -1 if the input must not be
added to corpus even if gives new coverage; and 0 otherwise; other values are
reserved for future use.

The `Fuzz` function must be in a package that `go-fuzz` can import. This means
the code you want to test can't be in package `main`.  Fuzzing `internal`
packages is supported, however.

In its basic form the Fuzz function just parses the input, and
go-fuzz ensures that it does not panic, crash the program, allocate insane
amount of memory nor hang. Fuzz function can also do application-level checks,
which will make testing more efficient (discover more bugs). For example,
Fuzz function can serialize all inputs that were successfully deserialized,
thus ensuring that serialization can handle everything deserialization can
produce. Or, Fuzz function can deserialize-serialize-deserialize-serialize
and check that results of first and second serialization are equal. Or, Fuzz
function can feed the input into two different implementations (e.g. dumb and
optimized) and check that the output is equal. To communicate application-level
bugs Fuzz function should panic (os.Exit(1) will work too, but panic message
contains more info). Note that Fuzz function should not output to stdout/stderr,
it will slow down fuzzing and nobody will see the output anyway. The exception
is printing info about a bug just before panicking.

Here is an example of a simple Fuzz function for image/png package:
```go
package png

import (
	"bytes"
	"image/png"
)

func Fuzz(data []byte) int {
	png.Decode(bytes.NewReader(data))
	return 0
}
```

A more useful Fuzz function would look like:
```go
func Fuzz(data []byte) int {
	img, err := png.Decode(bytes.NewReader(data))
	if err != nil {
		if img != nil {
			panic("img != nil on error")
		}
		return 0
	}
	var w bytes.Buffer
	err = png.Encode(&w, img)
	if err != nil {
		panic(err)
	}
	return 1
}
```

The second step is collection of initial input corpus. Ideally, files in the
corpus are as small as possible and as diverse as possible. You can use inputs
used by unit tests and/or generate them. For example, for an image decoding
package you can encode several small bitmaps (black, random noise, white with
few non-white pixels) with different levels of compressions and use that as the
initial corpus. Go-fuzz will deduplicate and minimize the inputs. So throwing in
a thousand of inputs is fine, diversity is more important.

Put the initial corpus into the workdir/corpus directory (in our case
```examples/png/corpus```). Go-fuzz will add own inputs to the corpus directory.
Consider committing the generated inputs to your source control system, this
will allow you to restart go-fuzz without losing previous work.

The [go-fuzz-corpus repository](https://github.com/dvyukov/go-fuzz-corpus) contains 
a bunch of examples of test functions and initial input corpuses for various packages.

The next step is to get go-fuzz:

```
$ go get -u github.com/dvyukov/go-fuzz/go-fuzz@latest github.com/dvyukov/go-fuzz/go-fuzz-build@latest
```

Then, download the corpus and build the test program with necessary instrumentation:
```
$ go get -d github.com/dvyukov/go-fuzz-corpus
$ cd $GOPATH/src/github.com/dvyukov/go-fuzz-corpus
$ cd png
$ go-fuzz-build
```
This will produce png-fuzz.zip archive.

Now we are ready to go:
```
$ go-fuzz
```

Go-fuzz will generate and test various inputs in an infinite loop. Workdir is
used to store persistent data like current corpus and crashers, it allows fuzzer
to continue after restart. Discovered bad inputs are stored in workdir/crashers
dir; where file without a suffix contains binary input, file with .quoted suffix
contains quoted input that can be directly copied into a reproducer program or a
test, file with .output suffix contains output of the test on this input. Every
few seconds go-fuzz prints logs to stderr of the form:
```
2015/04/25 12:39:53 workers: 500, corpus: 186 (42s ago), crashers: 3,
     restarts: 1/8027, execs: 12009519 (121224/sec), cover: 2746, uptime: 1m39s
```
Where ```workers``` means number of tests running in parallel (set with -procs
flag). ```corpus``` is current number of interesting inputs the fuzzer has
discovered, time in brackets says when the last interesting input was
discovered. ```crashers``` is number of discovered bugs (check out
workdir/crashers dir). ```restarts``` is the rate with which the fuzzer restarts
test processes. The rate should be close to 1/10000 (which is the planned
restart rate); if it is considerably higher than 1/10000, consider fixing already
discovered bugs which lead to frequent restarts. ```execs``` is total number of
test executions, and the number in brackets is the average speed of test
executions. ```cover``` is number of bits set in a hashed coverage bitmap, if this number
grows fuzzer uncovers new lines of code; size of the bitmap is 64K; ideally ```cover```
value should be less than ~5000, otherwise fuzzer can miss new interesting inputs
due to hash collisions. And finally ```uptime``` is uptime of the process. This same
information is also served via http (see the ```-http``` flag).

## Modules support

go-fuzz has preliminary support for fuzzing [Go Modules](https://github.com/golang/go/wiki/Modules). 
go-fuzz respects the standard `GO111MODULE` environment variable, which can be set to `on`, `off`, or `auto`. 

go-fuzz-build will add a `require` for `github.com/dvyukov/go-fuzz` to your go.mod. If desired, you may remove this once the build is complete.

Vendoring with modules is not yet supported. A `vendor` directory will be ignored, and go-fuzz will report an error if `GOFLAGS=-mod=vendor` is set.

Note that while modules are used to prepare the build, the final instrumented build is still done in GOPATH mode.
For most modules, this should not matter.

## libFuzzer support

go-fuzz-build can also generate an archive file
that can be used with [libFuzzer](https://llvm.org/docs/LibFuzzer.html)
instead of go-fuzz (requires linux).

Sample usage:

```
$ cd $GOPATH/src/github.com/dvyukov/go-fuzz-corpus/fmt
$ go-fuzz-build -libfuzzer  # produces fmt.a
$ clang -fsanitize=fuzzer fmt.a -o fmt.libfuzzer
$ ./fmt.libfuzzer
```

When run with `-libfuzzer`, go-fuzz-build adds the additional build tag
`gofuzz_libfuzzer` when building code.

## Continuous Fuzzing

Just as unit-testing, fuzzing is better done continuously.

Currently there are 2 services that offer continuous fuzzing based on go-fuzz:

- [gitlab.com](https://gitlab.com) ([tutorial](https://docs.gitlab.com/ee/user/application_security/coverage_fuzzing/))
- [fuzzbuzz.io](https://fuzzbuzz.io/) ([tutorial](https://docs.fuzzbuzz.io/getting-started/find-your-first-bug-in-go))

## Random Notes

go-fuzz-build builds the program with gofuzz build tag, this allows to put the
Fuzz function implementation directly into the tested package, but exclude it
from normal builds with ```// +build gofuzz``` directive.

If your inputs contain a checksum, it can make sense to append/update the checksum
in the ```Fuzz``` function. The chances that go-fuzz will generate the correct
checksum are very low, so most work will be in vain otherwise.

Go-fuzz can utilize several machines. To do this, start the coordinator process
separately:
```
$ go-fuzz -workdir=examples/png -coordinator=127.0.0.1:8745
```
It will manage persistent corpus and crashers and coordinate work of worker processes.
Then run one or more worker processes as:
```
$ go-fuzz -bin=./png-fuzz.zip -worker=127.0.0.1:8745 -procs=10
```

## External Articles

- [go-fuzz github.com/arolek/ase](https://medium.com/@dgryski/go-fuzz-github-com-arolek-ase-3c74d5a3150c): A step-by-step tutorial
- [DNS parser, meet Go fuzzer](https://blog.cloudflare.com/dns-parser-meet-go-fuzzer/): A success story with suggestions on how to write the ```Fuzz``` function
- [Automated Testing with go-fuzz](https://speakerdeck.com/filosottile/automated-testing-with-go-fuzz)
- [Going down the rabbit hole with go-fuzz](https://mijailovic.net/2017/07/29/go-fuzz/)
- [Fuzzing markdown parser with go-fuzz](https://blog.kowalczyk.info/article/n/fuzzing-markdown-parser-written-in-go.html)

## History rewrite

go-fuzz repository history was recently rewritten to exclude examples directory
to reduce total repository size and download time (see
[#88](https://github.com/dvyukov/go-fuzz/issues/88),
[#114](https://github.com/dvyukov/go-fuzz/issues/114) and
https://github.com/dvyukov/go-fuzz-corpus). Unfortunately, that means that
`go get -u` command will fail if you had a previous version installed.
Please remove $GOPATH/github.com/dvyukov/go-fuzz before running `go get` again.

## Credits and technical details

Go-fuzz fuzzing logic is heavily based on [american fuzzy lop](http://lcamtuf.coredump.cx/afl/),
so refer to [AFL readme](http://lcamtuf.coredump.cx/afl/README.txt) if you are
interested in technical details. AFL is written and maintained by
[Michal Zalewski](http://lcamtuf.coredump.cx/). Some of the mutations employed
by go-fuzz are inspired by work done by Mateusz Jurczyk, Gynvael Coldwind and
[Felix Gröbert](https://twitter.com/fel1x).

## Trophies

- [spec: non-integral constant can be converted to int](https://github.com/golang/go/issues/11350) **fixed**
- [cmd/compile: out of fixed registers](https://github.com/golang/go/issues/11352) **fixed**
- [cmd/compile: truncates constants](https://github.com/golang/go/issues/11326) **fixed**
- [cmd/compile: overflow in int -> string](https://github.com/golang/go/issues/11330) **fixed**
- [cmd/compile: bad HMUL](https://github.com/golang/go/issues/11358) **fixed**
- [cmd/compile: treecopy Name](https://github.com/golang/go/issues/11361) **fixed**
- [cmd/compile: accepts invalid identifiers](https://github.com/golang/go/issues/11359) **fixed**
- [cmd/compile: hangs compiling hex fp constant](https://github.com/golang/go/issues/11364) **fixed**
- [cmd/compile: mishandles int->complex conversion](https://github.com/golang/go/issues/11365) **fixed**
- [cmd/compile: allows to define blank methods on builtin types](https://github.com/golang/go/issues/11366) **fixed**
- [cmd/compile: mis-calculates a constant](https://github.com/golang/go/issues/11369) **fixed**
- [cmd/compile: interface conversion panic](https://github.com/golang/go/issues/11540) **fixed**
- [cmd/compile: nil pointer dereference](https://github.com/golang/go/issues/11588) **fixed**
- [cmd/compile: nil pointer dereference (2)](https://github.com/golang/go/issues/11666) **fixed**
- [cmd/compile: internal compiler error: plain block b3 len(Succs)==2, want 1](https://github.com/golang/go/issues/11589) **fixed**
- [cmd/compile: internal compiler error: b3.Succs has duplicate block b3](https://github.com/golang/go/issues/11593) **fixed**
- [cmd/compile: internal compiler error: newname nil](https://github.com/golang/go/issues/11610) **fixed**
- [cmd/compile: accepts invalid function type](https://github.com/golang/go/issues/11595) **fixed**
- [cmd/compile: internal compiler error: getinarg: not a func int](https://github.com/golang/go/issues/11596) **fixed**
- [cmd/compile: hangs converting int const to complex64](https://github.com/golang/go/issues/11597) **fixed**
- [cmd/compile: nil deref in error message](https://github.com/golang/go/issues/11614) **fixed**
- [cmd/compile: use of untyped nil in switch](https://github.com/golang/go/issues/11668) **fixed**
- [cmd/compile: implicitly converts complex constant to integer](https://github.com/golang/go/issues/11669) **fixed**
- [cmd/compile: assignment to entry in nil map](https://github.com/golang/go/issues/11670) **fixed**
- [cmd/compile: does not diagnose constant division by zero](https://github.com/golang/go/issues/11674) **fixed**
- [cmd/compile: does not detect a missing return](https://github.com/golang/go/issues/11698) **fixed**
- [cmd/compile: symbol ""._.args_stackmap listed multiple times](https://github.com/golang/go/issues/11699) **fixed**
- [cmd/compile: "0"[0] should not be a constant](https://github.com/golang/go/issues/11370) **fixed**
- [cmd/compile: unexpected %!(NOVERB)](https://github.com/golang/go/issues/13266) **fixed**
- [cmd/compile: wrong line number in error message](https://github.com/golang/go/issues/13267) **fixed**
- [cmd/compile: not-deterministic output](https://github.com/golang/go/issues/13268) **fixed**
- [cmd/compile: parsing problem](https://github.com/golang/go/issues/13270) **fixed**
- [cmd/compile: compiles incorrect program](https://github.com/golang/go/issues/13272) **fixed**
- [cmd/compile: does not compile correct program](https://github.com/golang/go/issues/13273)
- [cmd/compile: compiles incorrect program (2)](https://github.com/golang/go/issues/13274) **fixed**
- [cmd/compile: internal compiler error: want FUNC, but have int](https://github.com/golang/go/issues/17038) **fixed**
- [cmd/compile: nil deref](https://github.com/golang/go/issues/24447) **fixed**
- [cmd/asm: index out of range](https://github.com/golang/go/issues/11759) **fixed**
- [cmd/asm: index out of range (2)](https://github.com/golang/go/issues/12466) **fixed**
- [cmd/asm: index out of range (3)](https://github.com/golang/go/issues/12467) **fixed**
- [cmd/asm: index out of range (4)](https://github.com/golang/go/issues/12657) **fixed**
- [cmd/asm: slice bounds out of range](https://github.com/golang/go/issues/11760) **fixed**
- [cmd/asm: hang](https://github.com/golang/go/issues/11764) **fixed**
- [cmd/asm: hang (2)](https://github.com/golang/go/issues/12469) **fixed**
- [cmd/asm: hang (3)](https://github.com/golang/go/issues/12656) **fixed**
- [cmd/asm: nil deref](https://github.com/golang/go/issues/11765) **fixed**
- [cmd/asm: nil deref (2)](https://github.com/golang/go/issues/12468) **fixed**
- [cmd/asm: nil deref (3)](https://github.com/golang/go/issues/12614) **fixed**
- [cmd/asm: nil deref (4)](https://github.com/golang/go/issues/12627) **fixed**
- [cmd/asm: nil deref (5)](https://github.com/golang/go/issues/12655) **fixed**
- [cmd/asm: cannot happen: slice col](https://github.com/golang/go/issues/12654) **fixed**
- [cmd/asm: unactionable "invalid local variable type 0"](https://github.com/golang/go/issues/12658)
- [internal/trace: index out of range](https://github.com/golang/go/issues/11766) **fixed**
- [internal/trace: index out of range (2)](https://github.com/golang/go/issues/11769) **fixed**
- [internal/trace: nil deref](https://github.com/golang/go/issues/11767) **fixed**
- [internal/trace: nil deref (2)](https://github.com/golang/go/issues/11770) **fixed**
- [fmt: Printf loops on invalid verb spec](https://github.com/golang/go/issues/10674) **fixed**
- [fmt: incorrect overflow detection](https://github.com/golang/go/issues/10695) **fixed**
- [fmt: index out of range](https://github.com/golang/go/issues/10675) **fixed**
- [fmt: index out of range (2)](https://github.com/golang/go/issues/10745) **fixed**
- [fmt: index out of range (3)](https://github.com/golang/go/issues/10770) **fixed**
- [fmt: index out of range (4)](https://github.com/golang/go/issues/10771) **fixed**
- [fmt: index out of range (5)](https://github.com/golang/go/issues/10945) **fixed**
- [fmt: index out of range (6)](https://github.com/golang/go/issues/11376) **fixed**
- [regexp: slice bounds out of range](https://github.com/golang/go/issues/11176) **fixed**
- [regexp: slice bounds out of range (2)](https://github.com/golang/go/issues/11178) **fixed**
- [regexp: LiteralPrefix lies about completeness](https://github.com/golang/go/issues/11172)
- [regexp: LiteralPrefix lies about completeness (2)](https://github.com/golang/go/issues/11175)
- [regexp: POSIX regexp takes 4 seconds to execute](https://github.com/golang/go/issues/11181)
- [regexp: confusing behavior on invalid utf-8 sequences](https://github.com/golang/go/issues/11185)
- [regexp: considers "\Q\E*" as valid regexp](https://github.com/golang/go/issues/11187) **fixed**
- [time: allows signs for year/tz in format string](https://github.com/golang/go/issues/11128)
- [time: RFC3339 time.Parse can not parse string that come from time.Format](https://github.com/golang/go/issues/20555)
- [time: Parse panic: runtime error: index out of range](https://github.com/golang/go/issues/21113) **fixed**
- [math/big: incorrect string->Float conversion](https://github.com/golang/go/issues/11341) **fixed**
- [math/big: MakeFromLiteral with 0 mantissa and large exponent hangs](https://github.com/golang/go/issues/16176) **fixed**
- [net/http: can't send star request](https://github.com/golang/go/issues/11202) **fixed**
- [net/http: allows empty header names](https://github.com/golang/go/issues/11205) **fixed**
- [net/http: allows invalid characters in header values](https://github.com/golang/go/issues/11207) **fixed**
- [net/http: allows %-encoding after \[\]](https://github.com/golang/go/issues/11208) **fixed**
- [net/mail: ParseAddress/String corrupt address](https://github.com/golang/go/issues/11292) **fixed**
- [net/mail: parses invalid address](https://github.com/golang/go/issues/11293) **fixed**
- [net/mail: fails to escape address](https://github.com/golang/go/issues/11294) **fixed**
- [net/textproto: fails to trim header value](https://github.com/golang/go/issues/11204) **fixed**
- [archive/zip: cap out of range](https://github.com/golang/go/issues/10956) **fixed**
- [archive/zip: bad file size](https://github.com/golang/go/issues/10957) **fixed**
- [archive/zip: unexpected EOF](https://github.com/golang/go/issues/11144) **fixed**
- [archive/zip: file with wrong checksum is successfully decompressed](https://github.com/golang/go/issues/11146) **fixed**
- [archive/zip: unexpected EOF when reading archive](https://github.com/golang/go/issues/12449) **fixed**
- [archive/tar: slice bounds out of range](https://github.com/golang/go/issues/10959) **fixed**
- [archive/tar: slice bounds out of range (2)](https://github.com/golang/go/issues/10960) **fixed**
- [archive/tar: slice bounds out of range (3)](https://github.com/golang/go/issues/10966) **fixed**
- [archive/tar: slice bounds out of range (4)](https://github.com/golang/go/issues/10967) **fixed**
- [archive/tar: slice bounds out of range (5)](https://github.com/golang/go/issues/11167) **fixed**
- [archive/tar: deadly hang](https://github.com/golang/go/issues/10968) **fixed**
- [archive/tar: invalid memory address or nil pointer dereference](https://github.com/golang/go/issues/11168) **fixed**
- [archive/tar: invalid memory address or nil pointer dereference (2)](https://github.com/golang/go/issues/12435) **fixed**
- [archive/tar: Reader.Next returns nil header](https://github.com/golang/go/issues/11169) **fixed**
- [archive/tar: Writer incorrectly encodes header data](https://github.com/golang/go/issues/11171) **fixed**
- [archive/tar: incorrectly claims huge file size](https://github.com/golang/go/issues/12434)
- [archive/tar: reader returns bogus headers](https://github.com/golang/go/issues/12436) **fixed**
- [encoding/gob: panic: drop](https://github.com/golang/go/issues/10272) **fixed**
- [encoding/gob: makeslice: len out of range](https://github.com/golang/go/issues/10273) [3 bugs] **fixed**
- [encoding/gob: stack overflow](https://github.com/golang/go/issues/10415) **fixed**
- [encoding/gob: excessive memory consumption](https://github.com/golang/go/issues/10490) **fixed**
- [encoding/gob: decoding hangs](https://github.com/golang/go/issues/10491) **fixed**
- [encoding/gob: pointers to zero values are not initialized in Decode](https://github.com/golang/go/issues/11119)
- [encoding/gob: crash on malicious input](https://github.com/golang/go/issues/24447)
- [encoding/xml: allows invalid comments](https://github.com/golang/go/issues/11112)
- [encoding/json: detect circular data structures when encoding](https://github.com/golang/go/issues/10769)
- [encoding/asn1: index out of range](https://github.com/golang/go/issues/11129) **fixed**
- [encoding/asn1: incorrectly handles incorrect utf8 strings](https://github.com/golang/go/issues/11126) **fixed**
- [encoding/asn1: slice is lost during marshal/unmarshal](https://github.com/golang/go/issues/11130) **fixed**
- [encoding/asn1: call of reflect.Value.Type on zero Value](https://github.com/golang/go/issues/11127) **fixed**
- [encoding/asn1: Unmarshal accepts negative dates](https://github.com/golang/go/issues/11134) **fixed**
- [encoding/pem: can't decode encoded message](https://github.com/golang/go/issues/10980) **fixed**
- [crypto:x509: input not full blocks](https://github.com/golang/go/issues/11215) **fixed**
- [crypto/x509: division by zero](https://github.com/golang/go/issues/11233) **fixed**
- [image/jpeg: unreadByteStuffedByte call cannot be fulfilled](https://github.com/golang/go/issues/10387) **fixed**
- [image/jpeg: index out of range](https://github.com/golang/go/issues/10388) **fixed**
- [image/jpeg: invalid memory address or nil pointer dereference](https://github.com/golang/go/issues/10389) **fixed**
- [image/jpeg: Decode hangs](https://github.com/golang/go/issues/10413) **fixed**
- [image/jpeg: excessive memory usage](https://github.com/golang/go/issues/10532) **fixed**
- [image/png: slice bounds out of range](https://github.com/golang/go/issues/10414) **fixed**
- [image/png: slice bounds out of range (2)](https://github.com/golang/go/issues/12545) **fixed**
- [image/png: interface conversion: color.Color is color.NRGBA, not color.RGBA](https://github.com/golang/go/issues/10423) **fixed**
- [image/png: nil deref](https://github.com/golang/go/issues/10493) **fixed**
- [image/gif: image block is out of bounds](https://github.com/golang/go/issues/10676) **fixed**
- [image/gif: Decode returns an image with empty palette](https://github.com/golang/go/issues/11150) **fixed**
- [image/gif: LoopCount changes on round trip](https://github.com/golang/go/issues/11287) **fixed**
- [image/gif: Disposal is corrupted after round trip](https://github.com/golang/go/issues/11288)
- [image/gif: EOF instead of UnexpectedEOF](https://github.com/golang/go/issues/11390)
- [compress/flate: hang](https://github.com/golang/go/issues/10426) **fixed**
- [compress/lzw: compress/decompress corrupts data](https://github.com/golang/go/issues/11142) **fixed**
- [text/template: leaks goroutines on errors](https://github.com/golang/go/issues/10574#ref-issue-71873016)
- [text/template: Call using string as type int](https://github.com/golang/go/issues/10800) **fixed**
- [text/template: Call using complex128 as type string](https://github.com/golang/go/issues/10946) **fixed**
- [text/template: stack overflow](https://github.com/golang/go/issues/15618)
- [html/template: unidentified node type in allIdents](https://github.com/golang/go/issues/10610) **fixed**
- [html/template: unidentified node type in allIdents (2)](https://github.com/golang/go/issues/10801) **fixed**
- [html/template: unidentified node type in allIdents (3)](https://github.com/golang/go/issues/11118) **fixed**
- [html/template: unidentified node type in allIdents (4)](https://github.com/golang/go/issues/11356) **fixed**
- [html/template: escaping {{else}} is unimplemented](https://github.com/golang/go/issues/10611) **fixed**
- [html/template: runtime error: slice bounds out of range](https://github.com/golang/go/issues/10612) **fixed**
- [html/template: runtime error: slice bounds out of range (2)](https://github.com/golang/go/issues/10613) **fixed**
- [html/template: invalid memory address or nil pointer dereference](https://github.com/golang/go/issues/10615) **fixed**
- [html/template: panic: Call using zero Value argument](https://github.com/golang/go/issues/10634) **fixed**
- [html/template: nil pointer dereference](https://github.com/golang/go/issues/10673) **fixed**
- [html/template: slice bounds out of range](https://github.com/golang/go/issues/10799) **fixed**
- [mime: ParseMediaType parses invalid media types](https://github.com/golang/go/issues/11289) **fixed**
- [mime: Parse/Format corrupt parameters](https://github.com/golang/go/issues/11290) **fixed**
- [mime: Parse/Format corrupt parameters (2)](https://github.com/golang/go/issues/11291) **fixed**
- [go/constant: hang evaluating "-6e-1886451601"](https://github.com/golang/go/issues/20228) **fixed**
- [go/constant, math/big: panic while constructing constant "1i/1E-612198397"](https://github.com/golang/go/issues/20227)
- [go/scanner: accepts floating point literals with no decimals after E](https://github.com/golang/go/issues/17621)  **fixed**
- [go/parser: eats \r in comments](https://github.com/golang/go/issues/11151)
- [go/format: turns correct program into incorrect one](https://github.com/golang/go/issues/11274)
- [go/format: non-idempotent format](https://github.com/golang/go/issues/11275) **fixed**
- [go/format: adds }](https://github.com/golang/go/issues/11276) **fixed**
- [go/types: panics on invalid constant](https://github.com/golang/go/issues/11325) **fixed**
- [go/types: compiling hangs](https://github.com/golang/go/issues/11327) **fixed**
- [go/types: stupid shift](https://github.com/golang/go/issues/11328) **fixed**
- [go/types: line number out of range](https://github.com/golang/go/issues/11329)
- [go/types: assertion failed](https://github.com/golang/go/issues/11347) **fixed**
- [go/types: converts fp constant to string](https://github.com/golang/go/issues/11353) **fixed**
- [go/types: converts complex constant to string](https://github.com/golang/go/issues/11357) **fixed**
- [go/types: misses '-' in error message](https://github.com/golang/go/issues/11367) **fixed**
- [go/types: compiles invalid program with overflow](https://github.com/golang/go/issues/11368)
- [go/types: allows duplicate switch cases](https://github.com/golang/go/issues/11578) **fixed**
- [go/types: can shift complex numbers](https://github.com/golang/go/issues/11594) **fixed**
- [go/types: parses comma terminated fields](https://github.com/golang/go/issues/11611) **fixed**
- [go/types: int overflow in switch expression](https://github.com/golang/go/issues/11667) **fixed**
- [go/types: allows multiple-value in switch and case](https://github.com/golang/go/issues/11687) **fixed**
- [go/types: invalid error message for valid conversion to complex64](https://github.com/golang/go/issues/11590) **fixed**
- [debug/elf: index out of range](https://github.com/golang/go/issues/10996)
- [debug/elf: makeslice: len out of range](https://github.com/golang/go/issues/10997) **fixed**
- [debug/elf: slice bounds out of range](https://github.com/golang/go/issues/10999)
- [debug/pe: panic on interface conversion](https://github.com/golang/go/issues/30250)
- [debug/pe: slice bounds out of range](https://github.com/golang/go/issues/30253)
- [x/image/webp: index out of range](https://github.com/golang/go/issues/10383) **fixed**
- [x/image/webp: invalid memory address or nil pointer dereference](https://github.com/golang/go/issues/10384) **fixed**
- [x/image/webp: excessive memory consumption](https://github.com/golang/go/issues/10790)
- [x/image/webp: excessive memory consumption (2)](https://github.com/golang/go/issues/11395)
- [x/image/tiff: integer divide by zero](https://github.com/golang/go/issues/10393) **fixed**
- [x/image/tiff: index out of range](https://github.com/golang/go/issues/10394) **fixed**
- [x/image/tiff: slice bounds out of range](https://github.com/golang/go/issues/10395) **fixed**
- [x/image/tiff: index out of range](https://github.com/golang/go/issues/10597) **fixed**
- [x/image/tiff: slice bounds out of range](https://github.com/golang/go/issues/10596) **fixed**
- [x/image/tiff: integer divide by zero](https://github.com/golang/go/issues/10711) **fixed**
- [x/image/tiff: index out of range](https://github.com/golang/go/issues/10712) **fixed**
- [x/image/tiff: index out of range](https://github.com/golang/go/issues/11386)
- [x/image/tiff: excessive memory consumption](https://github.com/golang/go/issues/11389)
- [x/image/{tiff,bmp}: EOF instead of UnexpectedEOF](https://github.com/golang/go/issues/11391)
- [x/image/bmp: hang on degenerate image](https://github.com/golang/go/issues/10746) **fixed**
- [x/image/bmp: makeslice: len out of range](https://github.com/golang/go/issues/10396) **fixed**
- [x/image/bmp: out of memory](https://github.com/golang/go/issues/10399) **fixed**
- [x/net/icmp: runtime error: slice bounds out of range](https://github.com/golang/go/issues/10951)
- [x/net/html: void element <link> has child nodes](https://github.com/golang/go/issues/10535)
- [x/net/spdy: unexpected EOF](https://github.com/golang/go/issues/10539) **fixed**
- [x/net/spdy: EOF](https://github.com/golang/go/issues/10540) **fixed**
- [x/net/spdy: fatal error: runtime: out of memory](https://github.com/golang/go/issues/10542) **fixed**
- [x/net/spdy: stream id zero is disallowed](https://github.com/golang/go/issues/10543) **fixed**
- [x/net/spdy: processing of 35 bytes takes 7 seconds](https://github.com/golang/go/issues/10544) **fixed**
- [x/net/spdy: makemap: size out of range](https://github.com/golang/go/issues/10545) **fixed**
- [x/net/spdy: makeslice: len out of range](https://github.com/golang/go/issues/10547) **fixed**
- [x/crypto/ssh: Server panic on invalid input](https://github.com/golang/go/issues/11348) **fixed**
- [x/crypto/openpgp: ReadMessage(): Panic on invalid input in packet.nextSubpacket](https://github.com/golang/go/issues/11503) **fixed**
- [x/crypto/openpgp: ReadMessage(): Panic on invalid input in packet.PublicKeyV3.setFingerPrintAndKeyId](https://github.com/golang/go/issues/11504) **fixed**
- [x/crypto/openpgp: ReadMessage(): Panic on invalid input in math/big.nat.div](https://github.com/golang/go/issues/11505) **fixed**
- [gccgo: bogus index out of bounds](https://github.com/golang/go/issues/11522) **fixed**
- [gccgo: does not see stupidness of shift count](https://github.com/golang/go/issues/11524) **fixed**
- [gccgo: bogus integer constant overflow](https://github.com/golang/go/issues/11525) **fixed**
- [gccgo: segmentation fault](https://github.com/golang/go/issues/11526) **fixed**
- [gccgo: segmentation fault (2)](https://github.com/golang/go/issues/11536) **fixed**
- [gccgo: segmentation fault (3)](https://github.com/golang/go/issues/11558) **fixed**
- [gccgo: segmentation fault (4)](https://github.com/golang/go/issues/11559) **fixed**
- [gccgo: internal compiler error in set_type](https://github.com/golang/go/issues/11537) **fixed**
- [gccgo: internal compiler error in global_variable_set_init](https://github.com/golang/go/issues/11541) **fixed**
- [gccgo: internal compiler error: in wide_int_to_tree](https://github.com/golang/go/issues/11542) **fixed**
- [gccgo: internal compiler error in wide_int_to_tree (2)](https://github.com/golang/go/issues/12618) **fixed**
- [gccgo: internal compiler error in record_var_depends_on](https://github.com/golang/go/issues/11543) **fixed**
- [gccgo: internal compiler error in Builtin_call_expression](https://github.com/golang/go/issues/11544) **fixed**
- [gccgo: internal compiler error in check_bounds](https://github.com/golang/go/issues/11545) **fixed**
- [gccgo: internal compiler error in do_determine_type](https://github.com/golang/go/issues/11546) **fixed**
- [gccgo: internal compiler error in do_determine_type (2)](https://github.com/golang/go/issues/12937) **fixed**
- [gccgo: internal compiler error in backend_numeric_constant_expression](https://github.com/golang/go/issues/11548) **fixed**
- [gccgo: internal compiler error in type_size](https://github.com/golang/go/issues/11554) **fixed**
- [gccgo: internal compiler error in type_size (2)](https://github.com/golang/go/issues/11555) **fixed**
- [gccgo: internal compiler error in type_size (3)](https://github.com/golang/go/issues/11556) **fixed**
- [gccgo: internal compiler error in do_get_backend](https://github.com/golang/go/issues/11560) **fixed**
- [gccgo: internal compiler error in do_get_backend (2)](https://github.com/golang/go/issues/12325) **fixed**
- [gccgo: internal compiler error in do_get_backend (3)](https://github.com/golang/go/issues/12617) **fixed**
- [gccgo: internal compiler error in do_get_backend (4)](https://github.com/golang/go/issues/12939) **fixed**
- [gccgo: internal compiler error in create_tmp_var](https://github.com/golang/go/issues/11568) **fixed**
- [gccgo: internal compiler error in methods](https://github.com/golang/go/issues/11579) **fixed**
- [gccgo: internal compiler error in do_flatten](https://github.com/golang/go/issues/11592) **fixed**
- [gccgo: internal compiler error in do_flatten (2)](https://github.com/golang/go/issues/12319) **fixed**
- [gccgo: internal compiler error in do_flatten (3)](https://github.com/golang/go/issues/12320) **fixed**
- [gccgo: internal compiler error in declare_function](https://github.com/golang/go/issues/11557) **fixed**
- [gccgo: internal compiler error: in define](https://github.com/golang/go/issues/12316) **fixed**
- [gccgo: internal compiler error: in do_export](https://github.com/golang/go/issues/12321)
- [gccgo: internal compiler error in do_lower](https://github.com/golang/go/issues/12615) **fixed**
- [gccgo: internal compiler error in insert](https://github.com/golang/go/issues/12616) **fixed**
- [gccgo: internal compiler error in uniform_vector_p](https://github.com/golang/go/issues/12935) **fixed**
- [gccgo: accepts invalid UTF-8](https://github.com/golang/go/issues/11527) **fixed**
- [gccgo: spurious expected newline error](https://github.com/golang/go/issues/11528) **fixed**
- [gccgo: can apply ^ to true](https://github.com/golang/go/issues/11529) **fixed**
- [gccgo: hangs](https://github.com/golang/go/issues/11530) **fixed**
- [gccgo: hangs (2)](https://github.com/golang/go/issues/11531) **fixed**
- [gccgo: hangs (3)](https://github.com/golang/go/issues/11539) **fixed**
- [gccgo: rejects valid imaginary literal](https://github.com/golang/go/issues/11532) **fixed**
- [gccgo: rejects valid fp literal](https://github.com/golang/go/issues/11533) **fixed**
- [gccgo: accepts program with invalid identifier](https://github.com/golang/go/issues/11535) **fixed**
- [gccgo: accepts program with invalid identifier (2)](https://github.com/golang/go/issues/11547) **fixed**
- [gccgo: compiles weird construct](https://github.com/golang/go/issues/11561) **fixed**
- [gccgo: can do bitwise or on fp constants](https://github.com/golang/go/issues/11566) **fixed**
- [gccgo: treats nil as type](https://github.com/golang/go/issues/11567) **fixed**
- [gccgo: does not understand greek capiltal letter yot](https://github.com/golang/go/issues/11569) **fixed**
- [gccgo: does not understand CUNEIFORM SIGN DUG TIMES MI](https://github.com/golang/go/issues/12322) **fixed**
- [gccgo: allows to refer to builtin function not in call expression](https://github.com/golang/go/issues/11570) **fixed**
- [gccgo: bogus incompatible types in binary expression error](https://github.com/golang/go/issues/11572) **fixed**
- [gccgo: allows multiple definitions of a function](https://github.com/golang/go/issues/11573) **fixed**
- [gccgo: can shift by complex number](https://github.com/golang/go/issues/11574) **fixed**
- [gccgo: knowns unknown escape sequence](https://github.com/golang/go/issues/11575) **fixed**
- [gccgo: internal compiler error in start_function](https://github.com/golang/go/issues/11576) **fixed**
- [gccgo: internal compiler error: in start_function (2)](https://github.com/golang/go/issues/12324) **fixed**
- [gccgo: heap-buffer-overflow in Lex::skip_cpp_comment](https://github.com/golang/go/issues/11577) **fixed**
- [gccgo: does not convert untyped complex 0i to int in binary operation involving an int](https://github.com/golang/go/issues/11563)
- [gccgo: does not detect missing return](https://github.com/golang/go/issues/11591) **fixed**
- [gccgo: invalid error message for valid conversion to complex64](https://github.com/golang/go/issues/11615)
- [gccgo: can shift complex numbers](https://github.com/golang/go/issues/11616) **fixed**
- [gccgo: does not error on unused var](https://github.com/golang/go/issues/12317) **fixed**
- [gccgo: treats 0 as channel](https://github.com/golang/go/issues/12323) **fixed**
- [gccgo: does not recognize unused import](https://github.com/golang/go/issues/12326) **fixed**
- [gccgo: can shift by string](https://github.com/golang/go/issues/12936) **fixed**
- [github.com/golang/protobuf: call of reflect.Value.SetMapIndex on zero Value](https://github.com/golang/protobuf/issues/27) **fixed**
- [github.com/golang/protobuf: call of reflect.Value.Interface on zero Value in MarshalText](https://github.com/golang/protobuf/issues/33) **fixed**
- [github.com/golang/protobuf: Invalid map is successfully decoded](https://github.com/golang/protobuf/issues/34)
- [github.com/golang/protobuf: MarshalText incorrectly handles unknown bytes](https://github.com/golang/protobuf/issues/35)
- [github.com/golang/protobuf: MarshalText fails and prints to stderr](https://github.com/golang/protobuf/issues/36)
- [github.com/golang/protobuf: Unmarshaling errors for packed fields](https://github.com/golang/protobuf/issues/76) **fixed**
- [Equal prints to stderr and fails on what's handled by Marshal/Unmarshal](https://github.com/golang/protobuf/issues/114)
- [code.google.com/p/freetype-go: 42 crashers](https://code.google.com/p/freetype-go/issues/detail?id=17) [42 bugs]
- [github.com/cryptix/wav: 2 panics in header decoding](https://github.com/cryptix/wav/commit/2f49a0df0d213ee323f694e7bdee8b8a097dc698#diff-f86b763600291cbceee077a33133434a) **fixed**
- [github.com/spf13/hugo: 7 crashers](https://github.com/spf13/hugo/search?q=go-fuzz&type=Issues) **7 fixed**
- [github.com/Sereal/Sereal: 8 crashers](https://github.com/Sereal/Sereal/commit/c254cc3f2c48caffee6cd04ea8100a0150357a44) **fixed**
- [github.com/bradfitz/http2: Server.handleConn hangs](https://github.com/bradfitz/http2/issues/53) **fixed**
- [github.com/bradfitz/http2: nil pointer dereference in hpack.HuffmanDecode](https://github.com/bradfitz/http2/issues/56) **fixed**
- [github.com/bradfitz/http2: serverConn.readFrames goroutine leak](https://github.com/bradfitz/http2/issues/58)
- [github.com/golang/snappy: index out of range panic](https://github.com/golang/snappy/issues/11) **fixed**
- [github.com/bkaradzic/go-lz4: slice bounds out of range](https://github.com/bkaradzic/go-lz4/commit/b8d4dc7b31511bf5f39dfdb02d2ea7662eb8407c) **fixed**
- [github.com/kurin/blazer: string escape/unescape edge-cases, need to escape filename in DownloadFileByName()](https://github.com/kurin/blazer/issues/32) **fixed**
- [github.com/gocql/gocql: slice bounds out of range](https://github.com/gocql/gocql/commit/332853ab7b3c719dd67c657394139491c1f6deb7) **fixed**
- [github.com/gocql/gocql: slice bounds out of range](https://github.com/gocql/gocql/commit/58d90fab97daa2d9edd6e7a1b2a22bee8ce12c72) **fixed**
- [github.com/mdlayher/aoe: binary marshal/unmarshal inconsistency](https://github.com/mdlayher/aoe/commit/286c87727b95c9491e08cd909c93ac4a42218ee6) **fixed**
- [github.com/mdlayher/arp: slice bounds out of range](https://github.com/mdlayher/arp/commit/f237799bcd57ff8bb4773eb819a4b852875f01d0) **fixed**
- [github.com/mdlayher/ethernet: slice bounds out of range](https://github.com/mdlayher/ethernet/commit/da795f9b6b07b56d87722e5871e66519ac94f489) **fixed**
- [github.com/mdlayher/ndp: multiple crashers](https://github.com/mdlayher/ndp/commit/4356b107e450bdeae53069b7016d6342f8a4230d) **fixed**
- [github.com/mdlayher/netlink: slice bounds out of range](https://github.com/mdlayher/netlink/commit/1149baf1f02296bf9eac4a351d66861ecd7ed2a6) **fixed**
- [github.com/mdlayher/netlink: slice bounds out of range](https://github.com/mdlayher/netlink/commit/2861fca5d483a101d696320cf2bbf54424a886db) **fixed**
- [github.com/russross/blackfriday: index out of range panic in scanLinkRef](https://github.com/russross/blackfriday/issues/172) **fixed**
- [github.com/russross/blackfriday: index out of range panic in isReference](https://github.com/russross/blackfriday/issues/173) **fixed**
- [github.com/rwcarlsen/goexif: index out of range](https://github.com/rwcarlsen/goexif/issues/39)
- [github.com/tdewolff/minify: 8 crashers](https://github.com/tdewolff/minify/wiki) **fixed**
- [github.com/youtube/vitess/go/vt/sqlparser: index out of range](https://github.com/youtube/vitess/issues/767) **fixed**
- [github.com/youtube/vitess/go/vt/sqlparser: statement serialized incorrectly](https://github.com/youtube/vitess/issues/797) **fixed**
- [github.com/youtube/vitess/go/vt/sqlparser: statement serialized incorrectly (2)](https://github.com/youtube/vitess/issues/798)
- [gopkg.in/mgo.v2/bson: slice bounds out of range](https://github.com/go-mgo/mgo/issues/116) **fixed**
- [gopkg.in/mgo.v2/bson: Document is corrupted](https://github.com/go-mgo/mgo/issues/117) **fixed**
- [gopkg.in/mgo.v2/bson: Attempted to marshal empty Raw document](https://github.com/go-mgo/mgo/issues/120) **fixed**
- [cockroachdb/cockroach: crash on x % 0](https://github.com/cockroachdb/cockroach/pull/1799) **fixed**
- [cockroachdb/cockroach: panic when dealing with empty sql ident](https://github.com/cockroachdb/cockroach/pull/1808) **fixed**
- [cockroachdb/cockroach: parse literals more like Postgres](https://github.com/cockroachdb/cockroach/pull/1807) **fixed**
- [cockroachdb/cockroach: SELECT ("*") parse oddities](https://github.com/cockroachdb/cockroach/issues/1810) **fixed**
- [cockroachdb/cockroach: weird QualifiedName.Base panics on reproduce](https://github.com/cockroachdb/cockroach/issues/1938)
- [github.com/google/open-location-code: Extremely long codes can cause underflow errors](https://github.com/google/open-location-code/issues/12)
- [github.com/akrennmair/gopcap: incorrectly formed IP, UDP, TCP, ICMP packets can cause out of range errors](https://github.com/akrennmair/gopcap/commit/00e11033259acb75598ba416495bb708d864a010) **fixed**
- [github.com/gogo/protobuf: gogofast generates Unmarshal code that can panic](https://github.com/gogo/protobuf/issues/86) **fixed**
- [github.com/DHowett/go-plist: Various panics found through go-fuzz](https://github.com/DHowett/go-plist/issues/15)
- [github.com/streadway/amqp: go-fuzz fixes](https://github.com/streadway/amqp/pull/151)
- [github.com/andybalholm/cascadia: panic when parsing selectors like `:contains(`](https://github.com/andybalholm/cascadia/commit/3ad29d1ad1c4f2023e355603324348cf1f4b2d48) **fixed**
- [github.com/Azure/go-pkcs12: panic on malformed certificates](https://github.com/Azure/go-pkcs12/issues/25)
- [github.com/nats-io/gnatsd: panic on malformed input](https://github.com/nats-io/gnatsd/issues/95)
- [github.com/miekg/dns: 8 crashers](https://github.com/miekg/dns/pull/237) **fixed**
- [github.com/influxdb/influxdb: index out of range](https://github.com/influxdb/influxdb/pull/3570) **fixed**
- [collectd.org/network: 2 crashers](https://github.com/collectd/go-collectd/pull/6) **fixed**
- [collectd.org/network: index out of range](https://github.com/collectd/go-collectd/issues/10) **fixed**
- [github.com/arolek/ase: 2 crashers](https://github.com/arolek/ase/pull/18) **fixed**
- [github.com/lytics/confl: infinite loop on malformed input](https://github.com/lytics/confl/issues/6) **fixed**
- [github.com/zeebo/bencode: reject strings with negative length](https://github.com/zeebo/bencode/pull/15) **fixed**
- [github.com/hydrogen18/stalecucumber: 4 crashers](https://github.com/hydrogen18/stalecucumber/pull/5)
- [github.com/gonum/blas: cgo indexing error](https://github.com/gonum/blas/issues/133) **fixed**
- [OpenBLAS: incorrect idamax with NaN value](https://github.com/xianyi/OpenBLAS/issues/624)
- [github.com/eaburns/flac: 3 crashers](https://github.com/eaburns/flac/pull/6)
- [github.com/yvasiyarov/php_session_decoder: 4 crashers](https://github.com/yvasiyarov/php_session_decoder/pull/15)
- [xi2.org/x/xz: index out of bounds](https://github.com/xi2/xz/issues/3) **fixed**
- [github.com/pierrec/lz4: 2 crashers](https://github.com/pierrec/lz4/commit/0b67ae4bb1ab03691079e38dddbc3909d68de64f) **fixed**
- [github.com/dustin/go-coap: slice bounds out of range (1)](https://github.com/dustin/go-coap/commit/979f9a1787fc3091ba5c337a6d1d903432ce2007) **fixed**
- [github.com/dustin/go-coap: slice bounds out of range (2)](https://github.com/dustin/go-coap/commit/a2260b92ac405c9c63c4a89c15bb705a3f2928bf) **fixed**
- [github.com/dgryski/go-quicklz: many array-out-of-bounds issues](https://github.com/dgryski/go-quicklz/commit/6897f36a2bb707fe5b294fb9c06b7e086ab9503b) **fixed**
- [github.com/rasky/go-lzo: possible infinite loop with single byte input](https://github.com/rasky/go-lzo/commit/22d79fde8006d605b307e3d58b775d9c1f756d52) **fixed**
- [github.com/ulikunitz/xz: panic in lzma.writeRep](https://github.com/ulikunitz/xz/issues/3)
- [github.com/Preetam/sflow: excessive memory consumption](https://github.com/Preetam/sflow/issues/29) **fixed**
- [github.com/hashicorp/go-version: unhandled value out of range](https://github.com/hashicorp/go-version/pull/11) **fixed**
- [github.com/atlassian/gostatsd: Return an error instead of nil when parseline gets nil/empty input](https://github.com/atlassian/gostatsd/pull/5)
- [github.com/flynn/flynn/pkg/syslog/rfc5424: off-by-one](https://github.com/flynn/flynn/commit/a6bc68a86a36652b1d06d66052da67c1425faa2f) **fixed**
- [github.com/flynn/flynn/json5: decoder out of sync with scanner](https://github.com/flynn/json5/commit/29a96735397827a2923c9b669ee3188d601d9153) **fixed**
- [github.com/flynn/flynn/json5: broken carriage return parsing](https://github.com/flynn/json5/commit/7620272ed63390e979cf5882d2fa0506fe2a8db5) **fixed**
- [github.com/ipfs/go-ipfs: nil pointer deference in DHT RPC handler](https://github.com/ipfs/go-ipfs/pull/3200) **fixed**
- [github.com/buger/goreplay: fix panic in http headers parser function](https://github.com/buger/goreplay/pull/411) **fixed**
- [github.com/digitalocean/captainslog: incomplete timestamp caused panic](https://github.com/digitalocean/captainslog/pull/27/commits/d01c85621defb6fbbde22071de18f69bb3a74836) **fixed**
- [github.com/jlaffaye/ftp: panic: runtime error: index out of range](https://github.com/jlaffaye/ftp/issues/97) **fixed**
- [github.com/unidoc/unidoc: panic: interface conversion: pdf.PdfObject is nil, not *pdf.PdfObjectInteger](https://github.com/unidoc/unidoc/issues/77) **fixed**
- [github.com/unidoc/unidoc: panic: runtime error: invalid memory address or nil pointer dereference](https://github.com/unidoc/unidoc/issues/79) **fixed**
- [github.com/unidoc/unidoc: runtime: goroutine stack exceeds 1000000000-byte limit](https://github.com/unidoc/unidoc/issues/80) **fixed**
- [github.com/spenczar/tdigest: check slice bounds when unmarshaling](https://github.com/spenczar/tdigest/commit/91fdfcefc42381c63363418ad38d7d33233f79cd) **fixed**
- [github.com/spenczar/tdigest: check expected invariants while unmarshaling](https://github.com/spenczar/tdigest/commit/1ee0185dad51692b26b4ee6f9e111349f0cdb581) **fixed**
- [github.com/vcabbage/amqp: index out of range](https://github.com/vcabbage/amqp/commit/69a58fc911413722779226664a5a858b84758f94) **fixed**
- [github.com/gomarkdown/markdown: inifinite loop](https://github.com/gomarkdown/markdown/commit/5d96569c5a0d3cd46d961eddbb61e936e627774c) **fixed**
- [github.com/gomarkdown/markdown: inifinite loop](https://github.com/gomarkdown/markdown/commit/e0fc813169b926a2182bc6554888eb37d12261f7) **fixed**
- [github.com/gomarkdown/markdown: index out of range](https://github.com/gomarkdown/markdown/commit/5dd4b50fe81eda60f173e242ece05f24c5cc5cec) **fixed**
- [github.com/hajimehoshi/go-mp3: index out of range (1)](https://github.com/hajimehoshi/go-mp3/commit/22bc0be280079723dbb8e10295db01e925dc5640) **fixed**
- [github.com/hajimehoshi/go-mp3: index out of range (2)](https://github.com/hajimehoshi/go-mp3/commit/81bb838ef7ce492a3ea9d097a781ae1ed7f318b9) **fixed**
- [github.com/hajimehoshi/go-mp3: index out of range (3)](https://github.com/hajimehoshi/go-mp3/commit/3c185f92b8dbceefa913b64cae634ba47f452769) **fixed**
- [github.com/dhowden/tag: slice bounds out of range (1)](https://github.com/dhowden/tag/commit/737d3560ddb3e8e7319a5e9c494ccd749150d675) **fixed**
- [github.com/dhowden/tag: slice bounds out of range (2)](https://github.com/dhowden/tag/commit/1582ebc2a4525aeaccf7138ffd37e56f5117e49e) **fixed**
- [github.com/dhowden/tag: len out of range (3)](https://github.com/dhowden/tag/commit/d449289c5e6fec9ad6a68a9e850f22fe14fa7c97) **fixed**
- [github.com/dhowden/tag: slice bounds out of range (4)](https://github.com/dhowden/tag/commit/d2206af145611b630d612027486ffd9129bd3e09) **fixed**
- [github.com/tealeg/xlsx: slice bounds out of range (1)](https://github.com/tealeg/xlsx/commit/d40e2bb185733dd4bc3c4a1929c35ee844ed3379) **fixed**
- [github.com/hashicorp/hcl: crasher (logic error)](https://github.com/hashicorp/hcl/pull/239) **fixed**
- [github.com/hashicorp/hcl: crasher (off-by-one)](https://github.com/hashicorp/hcl/pull/240) **fixed**
- [github.com/hashicorp/hcl: format produces unparsable output (1)](https://github.com/hashicorp/hcl/pull/241) **fixed**
- [github.com/hashicorp/hcl: format produces unparsable output (2)](https://github.com/hashicorp/hcl/pull/243) **fixed**
- [github.com/hashicorp/hcl: format produces unparsable output (3)](https://github.com/hashicorp/hcl/pull/244) **fixed**
- [github.com/hashicorp/hcl: format produces unparsable output (4)](https://github.com/hashicorp/hcl/pull/245) **fixed**
- [github.com/francoispqt/gojay: panic on malformed JSON integers](https://github.com/francoispqt/gojay/issues/27) **fixed**
- [github.com/francoispqt/gojay: panic on malformed JSON floats](https://github.com/francoispqt/gojay/issues/32) **fixed**
- [github.com/eapache/go-xerial-snappy multiple panics with malformed inputs](https://github.com/eapache/go-xerial-snappy/commit/58803384a8be76cd0f84789b302c7b52d791d95f) **fixed**
- [github.com/trustelem/zxcvbn: multiple panics in password strength estimator](https://github.com/trustelem/zxcvbn/issues/1) **fixed**
- https://github.com/google/syzkaller: 6 crashers (
[1](https://github.com/google/syzkaller/commit/7c7ded697e6322b0975f061b7e268fe44f585dab),
[2](https://github.com/google/syzkaller/commit/3b37734422dc0cb40100287bbb3628d8d946c271),
[3](https://github.com/google/syzkaller/commit/f400a0da0fcd3e4d27d915b57c54f504813ef1d3),
[4](https://github.com/google/syzkaller/commit/967dc02d70f8e3d027738295977762cd4fbed5c7),
[5](https://github.com/google/syzkaller/commit/78b7ec0fbe23a5c674401123053d6372ea3ca9c6),
[6](https://github.com/google/syzkaller/commit/413e41473838fb74ccc081784afd6ddbbd44b797))
- [github.com/chai2010/guetzli-go: index out of range](https://github.com/chai2010/guetzli-go/issues/11)
- [github.com/pixiv/go-libjpeg: segmentation violation (1)](https://github.com/pixiv/go-libjpeg/issues/51) **fixed**
- [github.com/pixiv/go-libjpeg: segmentation violation (2)](https://github.com/pixiv/go-libjpeg/issues/58)
- [github.com/pixiv/go-libjpeg: panic on encoding after decoding](https://github.com/pixiv/go-libjpeg/issues/55) **fixed**
- [github.com/z7zmey/php-parser: index out of range and nil pointer dereference](https://github.com/z7zmey/php-parser/issues/98)
- [github.com/uber/makisu: index out of range (1)](https://github.com/uber/makisu/issues/266) **fixed**
- [github.com/uber/makisu: index out of range (2)](https://github.com/uber/makisu/issues/271)
- [github.com/google/go-attestation: out of memory](https://github.com/google/go-attestation/issues/126) **fixed**
- [github.com/buger/jsonparser index out of range](https://github.com/buger/jsonparser/issues/178)
- [github.com/buger/jsonparser infinite loop](https://github.com/buger/jsonparser/issues/179)
- [github.com/hjson/hjson-go: panic on nil](https://github.com/hjson/hjson-go/issues/16) **fixed**
- [github.com/hjson/hjson-go: panic on invalid syntax](https://github.com/hjson/hjson-go/issues/17) **fixed**
- [github.com/google/gofuzz: off-by-one error](https://github.com/google/gofuzz/issues/46) **fixed**
- [github.com/bookingcom/nanotube: index out of range](https://github.com/bookingcom/nanotube/pull/97) **fixed**
- [github.com/ProtonMail/crypto: panic on fingerpring subpacket](https://github.com/ProtonMail/crypto/pull/66) **fixed**
- [github.com/robfig/cron: panic on malformed schedule string](https://github.com/robfig/cron/pull/365)
- [github.com/cronokirby/saferith: infinite loop in ModSqrt](https://github.com/cronokirby/saferith/commit/d39f5a274f7c3b8c8d60456b1525cce26ffacfe7) **fixed**
- [github.com/go-git/go-git: infinite loop in revision parser](https://github.com/go-git/go-git/pull/475)

**If you find some bugs with go-fuzz and are comfortable with sharing them, I would like to add them to this list.** Please either send a pull request for README.md (preferable) or file an issue. If the source code is closed, you can say just "found N bugs in project X". Thank you.
