/* wmudmount
 * Copyright © 2010-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"
#include "misc.h"

#include <stdlib.h>

#ifdef HAVE_PRCTL
#include <sys/prctl.h>
#else
#include <sys/resources.h>
#endif

#include <glib.h>

#ifdef HAVE_GNOME_KEYRING
#include <gnome-keyring.h>
#include <gnome-keyring-memory.h>

static GnomeKeyringPasswordSchema schema = {
    .item_type = GNOME_KEYRING_ITEM_GENERIC_SECRET,
    .attributes = {
        { "device-type", GNOME_KEYRING_ATTRIBUTE_TYPE_STRING },
        { "device-uuid", GNOME_KEYRING_ATTRIBUTE_TYPE_STRING },
        { NULL, 0 }
    }
};
#endif

#include "die.h"
#include "askpass.h"
#include "secureentry.h"

gboolean allow_core_files = FALSE;
gboolean allow_insecure_memory = FALSE;
static gboolean use_keyring = FALSE;

/****************** Option parsing ******************/

void init_keyring() {
    warn(DEBUG_DEBUG, "Disabling core files...");
    errno=0;
#ifdef HAVE_PRCTL
    if (prctl(PR_SET_DUMPABLE, 0)) {
#else
    struct rlimit r;
    r.rlim_cur=r.rlim_max=0;
    if (setrlimit(RLIMIT_CORE, &r)) {
#endif
        if (allow_core_files) {
            die("Could not disable core files: %s", strerror(errno));
        } else {
            warn(DEBUG_WARN, "Could not disable core files: %s", strerror(errno));
        }
    } else {
        warn(DEBUG_INFO, "Core files disabled");
    }

    if (!allow_insecure_memory) {
        warn(DEBUG_DEBUG, "Checking whether secure memory can be allocated...");
#ifdef HAVE_GNOME_KEYRING
        gpointer p = gnome_keyring_memory_try_realloc(NULL, 1024);
        if (!p) {
            die("Could not allocate 1024 bytes of secure memory (as a test)");
        }
        warn(DEBUG_INFO, "Allocated at %p", p);
        gnome_keyring_memory_free(p);
        warn(DEBUG_INFO, "Secure memory can be allocated");
#else
        die("Support for secure memory via gnome-keyring was not compiled in. Specify\n--allow-insecure-memory on the command line to proceed.");
#endif
    }

#ifdef HAVE_GNOME_KEYRING
    if (gnome_keyring_is_available()) {
        warn(DEBUG_INFO, "Keyring will be used for crypto device passwords");
        use_keyring = TRUE;
    } else
#endif
    {
        warn(DEBUG_INFO, "Keyring is not available, crypto device passwords will NOT be saved");
        use_keyring = FALSE;
    }
}

static void save_pass_to_keyring(wmudisks_volume *v, gboolean session, const char *pass) {
    char *name = g_strdup_printf("%s crypto device passphrase for UUID %s", v->crypto_device_type, v->crypto_device_uuid);
#ifdef HAVE_GNOME_KEYRING
    if (use_keyring) {
        GnomeKeyringResult ok = gnome_keyring_store_password_sync(
            &schema, session ? GNOME_KEYRING_SESSION : GNOME_KEYRING_DEFAULT, name, pass,
            "device-type", v->crypto_device_type,
            "device-uuid", v->crypto_device_uuid,
            NULL
        );
        if (ok != GNOME_KEYRING_RESULT_OK) {
            const char *err = (ok == GNOME_KEYRING_RESULT_CANCELLED) ? "Cancelled by user" : gnome_keyring_result_to_message(ok);
            warn(DEBUG_ERROR, "Could not save %s: %s", name, err);
        }
    } else
#endif
    {
        warn(DEBUG_ERROR, "Could not save %s: no keyring is available", name);
    }
    g_free(name);
}

void forget_pass_from_keyring(wmudisks_volume *v) {
#ifdef HAVE_GNOME_KEYRING
    if (use_keyring) {
        GnomeKeyringResult ok = gnome_keyring_delete_password_sync(&schema,
            "device-type", v->crypto_device_type,
            "device-uuid", v->crypto_device_uuid,
            NULL
        );
        if (ok != GNOME_KEYRING_RESULT_OK && ok != GNOME_KEYRING_RESULT_NO_MATCH) {
            const char *err = (ok == GNOME_KEYRING_RESULT_CANCELLED) ? "Cancelled by user" : gnome_keyring_result_to_message(ok);
            warn(DEBUG_ERROR, "Could not delete %s crypto device passphrase for UUID %s: %s", v->crypto_device_type, v->crypto_device_uuid, err);
        }
    }
#endif
}

char *load_pass_from_keyring(wmudisks_volume *v) {
    gchar *pass = NULL;

#ifdef HAVE_GNOME_KEYRING
    if (use_keyring) {
        GnomeKeyringResult ok = gnome_keyring_find_password_sync(&schema, &pass,
            "device-type", v->crypto_device_type,
            "device-uuid", v->crypto_device_uuid,
            NULL
        );
        if (ok != GNOME_KEYRING_RESULT_OK && ok != GNOME_KEYRING_RESULT_NO_MATCH) {
            const char *err = (ok == GNOME_KEYRING_RESULT_CANCELLED) ? "Cancelled by user" : gnome_keyring_result_to_message(ok);
            warn(DEBUG_ERROR, "Could not load %s crypto device passphrase for UUID %s: %s", v->crypto_device_type, v->crypto_device_uuid, err);
            return NULL;
        }
        return pass;
    }
#endif
    return NULL;
}

void free_pass(char *pass) {
#ifdef HAVE_GNOME_KEYRING
    gnome_keyring_free_password(pass);
#else
    free(pass);
#endif
}

char *ask_for_pass(wmudisks_volume *v, GtkWidget *transient_for, GError **err) {
    g_warn_if_fail(err == NULL || *err == NULL);

    GtkWidget *dialog = gtk_dialog_new_with_buttons("Unlock device",
        GTK_WINDOW(transient_for),
        GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
        "_Unlock", GTK_RESPONSE_ACCEPT,
        "_Cancel", GTK_RESPONSE_CANCEL,
        NULL
    );
    gtk_dialog_set_default_response((GtkDialog*)dialog, GTK_RESPONSE_ACCEPT);
    gtk_window_set_icon_name((GtkWindow*)dialog, "dialog-password");

    GtkBox *content = (GtkBox*)gtk_dialog_get_content_area((GtkDialog*)dialog);

    GtkBox *hbox = (GtkBox*)gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 10);
    gtk_box_pack_start(content, (GtkWidget*)hbox, FALSE, FALSE, 10);
    GtkWidget *image = gtk_image_new_from_gicon(v->icon, GTK_ICON_SIZE_DIALOG);
    gtk_misc_set_alignment((GtkMisc*)image, 0.5, 0.0);
    gtk_box_pack_start(hbox, image, FALSE, FALSE, 0);
    GtkWidget *label = gtk_label_new(NULL);
    char *txt = g_markup_printf_escaped("<b><big>%s</big></b>", v->desc);
    gtk_label_set_markup((GtkLabel*)label, txt);
    g_free(txt);
    gtk_misc_set_alignment((GtkMisc*)label, 0.0, 0.5);
    gtk_label_set_line_wrap((GtkLabel*)label, TRUE);
    gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);

    label = gtk_label_new("This device is password protected. To make it available for use, enter the passphrase here.");
    gtk_misc_set_alignment((GtkMisc*)label, 0.0, 0.5);
    gtk_label_set_line_wrap((GtkLabel*)label, TRUE);
    gtk_box_pack_start(content, label, FALSE, FALSE, 0);

    hbox = (GtkBox*)gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 10);
    gtk_box_pack_start(content, (GtkWidget*)hbox, FALSE, FALSE, 0);
    label = gtk_label_new_with_mnemonic("_Passphrase:");
    gtk_misc_set_alignment((GtkMisc*)label, 1.0, 0.5);
    gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);
    GtkEntryBuffer *entrybuf = secure_entry_buffer_new();
    secure_entry_buffer_set_allow_insecure_memory(entrybuf, allow_insecure_memory);
    GtkWidget *entry = gtk_entry_new_with_buffer(entrybuf);
    gtk_entry_set_visibility((GtkEntry*)entry, FALSE);
    gtk_entry_set_activates_default((GtkEntry*)entry, TRUE);
    gtk_box_pack_start(hbox, entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget((GtkLabel*)label, entry);

    GtkWidget *rb_forget = gtk_radio_button_new_with_mnemonic(NULL, "_Forget passphrase immediately");
    GtkWidget *rb_remember = gtk_radio_button_new_with_mnemonic_from_widget((GtkRadioButton*)rb_forget, "_Remember passphrase");
    GtkWidget *rb_session = gtk_radio_button_new_with_mnemonic_from_widget((GtkRadioButton*)rb_forget, "Remember passphrase until _logout");

    gtk_toggle_button_set_active((GtkToggleButton*)rb_forget,TRUE);
    gtk_widget_set_sensitive(rb_remember, use_keyring);
    gtk_widget_set_sensitive(rb_session, use_keyring);

    gtk_box_pack_start(content, rb_forget, FALSE, FALSE, 0);
    gtk_box_pack_start(content, rb_remember, FALSE, FALSE, 0);
    gtk_box_pack_start(content, rb_session, FALSE, FALSE, 0);

    gtk_widget_show_all((GtkWidget*)content);
    int response = gtk_dialog_run((GtkDialog*)dialog);
    char *pass = NULL;
    switch (response) {
      case GTK_RESPONSE_ACCEPT:
        pass = (char *)gtk_entry_get_text((GtkEntry*)entry);
        if (gtk_toggle_button_get_active((GtkToggleButton*)rb_remember)) {
            save_pass_to_keyring(v, FALSE, pass);
        } else if (gtk_toggle_button_get_active((GtkToggleButton*)rb_session)) {
            save_pass_to_keyring(v, TRUE, pass);
        } else {
            forget_pass_from_keyring(v);
        }
        break;
      case GTK_RESPONSE_CANCEL:
        g_set_error(err, APP_GENERIC_ERROR, 0, "Passphrase request cancelled by user");
        break;
      default:
        g_set_error(err, APP_GENERIC_ERROR, 0, "Passphrase request failed");
        break;
    }
    gtk_widget_destroy(dialog);
    return pass;
}
