#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
//
//
# include <mntent.h>




static void *
private_is_mounted (void *p) {
    if(!p) {
        DBG ("fstab.i:private_is_mounted() mnt_point != NULL not met!\n");
        return NULL;
    }
    const gchar *mnt_fsname = p;
    gchar *mnt_point = realpath((gchar *)p, NULL);
    
    struct mntent *m;
    //const gchar *mnttab;
    FILE *tab_file;

    // try both /etc/mtab and /proc/mounts 
    const gchar *mfile[]={"/proc/mounts", "/etc/mtab", NULL};
    const gchar **pfile;
    for (pfile=mfile; pfile && *pfile; pfile++){
	if((tab_file = fopen (*pfile, "r")) == NULL) {
	    continue;
	}
	fclose(tab_file);
	tab_file = setmntent (*pfile, "r");

	if(!tab_file) {
	    perror ("setmntent:");
	    g_free(mnt_point);
	    return NULL;
	}
	struct mntent mntbuf;
	gchar buf[2048]; 
        while ((m = getmntent_r (tab_file, &mntbuf, buf, 2048)) != NULL) {	
	    NOOP("private_is_mounted():%s:  %s  or  %s\n", mnt_point, m->mnt_dir, m->mnt_fsname);
	    if((mnt_point && strcmp (m->mnt_dir, mnt_point) == 0) || 
	       (mnt_fsname && strcmp (m->mnt_fsname, mnt_fsname) == 0)) {
		NOOP("private_is_mounted(): GOTCHA  %s  %s:%s\n", m->mnt_dir, m->mnt_fsname, mnt_point);
		endmntent (tab_file);
		g_free(mnt_point);
		return GINT_TO_POINTER (1);
	    }
	}
	endmntent (tab_file);
    }
    g_free(mnt_point);
    return NULL;
}
static gint
my_strcmp (gconstpointer a, gconstpointer b) {
    NOOP ("FSTAB:     compare %s <=> %s\n", (char *)a, (char *)b);
    return (strcmp ((char *)a, (char *)b));
}

static GList *
destroy_fstab_list (GList ** fstab_list_p) {
    GList *p;
    for(p = *fstab_list_p; p && p->data; p = p->next) {
        g_free (p->data);
    }
    g_list_free (*fstab_list_p);
    return NULL;
}

static gboolean
include_in_xfdir (struct mntent *mnt_struct, GList ** fstab_list_p) {
    /* skip swap */
    if(strstr (mnt_struct->mnt_type, MNTTYPE_SWAP))
        return FALSE;
    /* skip missing mount points */
    if(!rfm_g_file_test (mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
        return FALSE;
    if(g_list_find_custom (*fstab_list_p, mnt_struct->mnt_dir, my_strcmp) == NULL) {
        *fstab_list_p = g_list_append (*fstab_list_p, g_strdup (mnt_struct->mnt_dir));
        NOOP ("FSTAB: adding %s\n", mnt_struct->mnt_dir);
        return TRUE;
    } else {
        NOOP ("FSTAB: skipping %s\n", mnt_struct->mnt_dir);
        return FALSE;
    }
    return TRUE;
}

static int
count_elements (void
    ) {

    GList *fstab_list = NULL;
    struct mntent *mnt_struct;
    FILE *fstab_fd;
    int i = 0;
    char *files[] = { "/etc/fstab", "/etc/mtab", NULL };
    char **p;
    for(p = files; *p; p++) {
        if((fstab_fd = setmntent (*p, "r")) == NULL) {
            DBG ("Unable to open %s\n", "/etc/fstab");
            return (0);
        }
	struct mntent mntbuf;
	gchar buf[2048]; 
        while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
            if(!include_in_xfdir (mnt_struct, &fstab_list))
                continue;
            i++;
        }
        (void)endmntent (fstab_fd);
    }
    fstab_list = destroy_fstab_list (&fstab_list);
    return i;
}

static gboolean
is_user_type (const gchar * mnt_point) {
    FILE *fstab_fd;
    if((fstab_fd = setmntent ("/etc/fstab", "r")) == NULL) {
        DBG ("Unable to open %s\n", "/etc/fstab");
        return FALSE;
    }
    struct mntent *mnt_struct;
    gboolean user_type = FALSE;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strstr (mnt_struct->mnt_type, MNTTYPE_SWAP))
            continue;
        if(!rfm_g_file_test (mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
            continue;
        if(strcmp (mnt_point, mnt_struct->mnt_dir) == 0 && strstr (mnt_struct->mnt_opts, "user")) {
            user_type = TRUE;
            break;
        }
    }
    (void)endmntent (fstab_fd);
    return user_type;
}

static void
get_mnt_items (xfdir_t * xfdir_p, gint first) {

    GList *fstab_list = NULL;
    char *files[] = { "/etc/fstab", "/etc/mtab", NULL };
    char **p;
    int i = first;
    struct mntent *mnt_struct;
    for(p = files; *p; p++) {
        NOOP ("FSTAB:  parsing %s\n", *p);
        FILE *fstab_fd = setmntent (*p, "r");

        struct mntent mntbuf;
	gchar buf[2048]; 
        while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
            if(!include_in_xfdir (mnt_struct, &fstab_list))
                continue;

            xfdir_p->gl[i].en = rfm_stat_entry (mnt_struct->mnt_dir, 0);
            if(strcmp (*p, "/etc/mtab") == 0) {
                NOOP ("MTAB: setting MTAB type for %s\n", xfdir_p->gl[i].en->path);
                SET_MTAB_TYPE (xfdir_p->gl[i].en->type);
            } else {
                xfdir_p->gl[i].en->tag = g_strdup (mnt_struct->mnt_fsname);
            }
            xfdir_p->gl[i].pathv = g_strdup (mnt_struct->mnt_dir);
            NOOP ("FSTAB:  adding item %d %s \n", i, xfdir_p->gl[i].pathv);

            SET_FSTAB_TYPE (xfdir_p->gl[i].en->type);
            if(strstr (mnt_struct->mnt_opts, "user")) {
                SET_USER_TYPE (xfdir_p->gl[i].en->type);
            }
            /* set type */
            NOOP ("FSTAB:   checking %s for fstab type\n", mnt_struct->mnt_type);
            int type = get_fstab_type (mnt_struct->mnt_type);
            switch (type) {
            case __NFS_TYPE:
                SET_NFS_TYPE (xfdir_p->gl[i].en->type);
                break;
            case __CDFS_TYPE:
                SET_CDFS_TYPE (xfdir_p->gl[i].en->type);
                break;
            case __PROC_TYPE:
                SET_PROC_TYPE (xfdir_p->gl[i].en->type);
                break;
            case __SMB_TYPE:
                SET_SMB_TYPE (xfdir_p->gl[i].en->type);
                break;
            default:
                break;
            }
            i++;
        }
        (void)endmntent (fstab_fd);
    }
    fstab_list = destroy_fstab_list (&fstab_list);

}

static gchar *
fstab_get_mnt_dir (gchar * mnt_fsname) {
    FILE *fstab_fd = setmntent ("/etc/mtab", "r");
    if(!fstab_fd)
        return NULL;
    struct mntent *mnt_struct;
    gchar *mnt_dir = NULL;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (mnt_fsname, mnt_struct->mnt_fsname) == 0) {
            // hit: multiple entries use first listed 
            // user types have preference and use last listed 
            if(strstr (mnt_struct->mnt_opts, "user")) {
                g_free (mnt_dir);
                mnt_dir = g_strdup (mnt_struct->mnt_dir);
            }
            if(!mnt_dir) {
                mnt_dir = g_strdup (mnt_struct->mnt_dir);
            }
        }
    }
    (void)endmntent (fstab_fd);
    return mnt_dir;
}

static  gchar *
fstab_get_mnt_fsname (gchar * mnt_dir) {
    FILE *fstab_fd = setmntent ("/etc/mtab", "r");
    if(!fstab_fd)
        return NULL;
    struct mntent *mnt_struct;
    gchar *mnt_fsname = NULL;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (mnt_dir, mnt_struct->mnt_dir) == 0) {
            // hit: multiple entries use first listed 
            // user types have preference and use last listed 
            if(strstr (mnt_struct->mnt_opts, "user")) {
                g_free (mnt_fsname);
                mnt_fsname = g_strdup (mnt_struct->mnt_fsname);
            }
            if(!mnt_fsname) {
                mnt_fsname = g_strdup (mnt_struct->mnt_fsname);
            }
        }
    }
    (void)endmntent (fstab_fd);
    return mnt_fsname;
}

static void
set_mounts_info (record_entry_t * en) {
    FILE *fstab_fd = setmntent ("/etc/mtab", "r");
    if(!fstab_fd) return;
    gchar *partition = en->path;
    struct mntent *mnt_struct;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (partition, mnt_struct->mnt_fsname) == 0) {
            NOOP ("MOUNT: %s == %s\n", partition, mnt_struct->mnt_fsname);
            // hit: multiple entries use last listed 
	    g_free (en->tag);
	    en->tag = g_strdup (mnt_struct->mnt_dir);
	}
    }
    (void)endmntent (fstab_fd);
    return;
}
#if 10
static void
set_partition_info (record_entry_t * en) {
    FILE *fstab_fd = setmntent ("/etc/fstab", "r");
    if(!fstab_fd)
        return;
    gchar *partition = en->path;
    struct mntent *mnt_struct;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (partition, mnt_struct->mnt_fsname) == 0) {
            NOOP ("MOUNT: %s == %s\n", partition, mnt_struct->mnt_fsname);
            // hit: multiple entries use last listed if not user type 
            if(strstr (mnt_struct->mnt_opts, "user")) {
                SET_USER_TYPE (en->type);
                g_free (en->tag);
                en->tag = g_strdup (mnt_struct->mnt_dir);
            } else {
                if(!IS_USER_TYPE (en->type)) {
                    g_free (en->tag);
                    en->tag = g_strdup (mnt_struct->mnt_dir);
                }
            }
        }
    }
    (void)endmntent (fstab_fd);
    return;
}
#endif

static int
partition_items (xfdir_t * xfdir_p, int i, gboolean full_read) {
    FILE *partitions = fopen ("/proc/partitions", "r");
    if(partitions) {
        char line[1024];
        memset (line, 0, 1024);
        while(fgets (line, 1023, partitions) && !feof (partitions)) {
            if(strlen (line) < 5)
                continue;
            if(strchr (line, '#'))
                continue;
            NOOP ("FSTAB: %s\n", line);
            gchar *p = strrchr (line, ' ');
            if(p) {
                NOOP ("FSTAB: add --> %s\n", p);
                g_strstrip (p);
                //if (strlen(p) && isdigit(p[strlen(p)-1])) {
                if(strlen (p)) {
                    if(full_read) {
                        xfdir_p->gl[i].pathv = g_strdup_printf ("/dev/%s", p);
                        xfdir_p->gl[i].en = rfm_stat_entry (xfdir_p->gl[i].pathv, 0);
                        NOOP ("FSTAB:  adding item %d %s \n", i, p);
                        SET_PARTITION_TYPE (xfdir_p->gl[i].en->type);
			SET_SBLK(xfdir_p->gl[i].en->type);
                        SET_LOCAL_TYPE (xfdir_p->gl[i].en->type);
                        set_partition_info (xfdir_p->gl[i].en);
                    }
                    i++;
                }
            }
            memset (line, 0, 1024);
        }
        fclose (partitions);
    }
    return i;
}

static int
count_partitions (void) {
    return partition_items (NULL, 0, FALSE);
}

static int
get_partition_items (xfdir_t * xfdir_p, int i) {
    return partition_items (xfdir_p, i, TRUE);
}

static xfdir_t *
private_get_xfdir (xfdir_t * xfdir_p) {

    gint elements = count_elements ();
    gint partitions = count_partitions ();

    gint first = malloc_items(xfdir_p, partitions + elements);
    // g_error taken care of within function
    // fstab and mtab:
    get_mnt_items (xfdir_p, first);
    // partitions:
    get_partition_items (xfdir_p, elements + first);

    NOOP (stderr, "FSTAB: gl is 0x%x with count=%ld\n", GPOINTER_TO_INT(xfdir_p->gl), (long)xfdir_p->pathc);

    return (xfdir_p);
}

/* 
 * This should return the fstab type, -1 if default or NULL is not in fstab...
 * Fstab types *are* (or should be) supported in constants.h.
 */
static void *
fstab_is_in_fstab (void *p) {
    struct mntent *mnt_struct;
    FILE *fstab_fd;
    int result = 0;
    const gchar *path = (const gchar *)p;
    if((fstab_fd = setmntent ("/etc/fstab", "r")) == NULL) {
        DBG ("Unable to open %s\n", "/etc/fstab");
        return (NULL);
    }

    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strstr (mnt_struct->mnt_type, MNTTYPE_SWAP))
            continue;
        if(!rfm_g_file_test (mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
            continue;

        NOOP ("fstab_is_in_fstab():%s --->  %s   or   %s\n", 
		path, mnt_struct->mnt_dir, mnt_struct->mnt_fsname);

        if(strcmp (path, mnt_struct->mnt_dir) == 0 || strcmp (path, mnt_struct->mnt_fsname) == 0) {
            result = get_fstab_type (mnt_struct->mnt_type);
            NOOP ("fstab_is_in_fstab():%s ---> %d %s\n", 
		    mnt_struct->mnt_fsname, result, mnt_struct->mnt_type);
            break;
        }
    }

    (void)endmntent (fstab_fd);
    return GINT_TO_POINTER (result);
}

static gchar *
fstab_df (void) {
    gchar *df_string = NULL;
    int line_count = 0;
    char line[2048];
    // This is for Linux...*/
    gchar *files[] = { "/proc/partitions", "/proc/mounts", "/etc/mtab", NULL };
    gchar **p;
    for(p = files; p && *p; p++) {
        FILE *partitions = fopen (*p, "r");
        if(partitions) {
            while(fgets (line, 2047, partitions) && !feof (partitions)) {
                line_count++;
            }
            fclose (partitions);
        }
    }
    df_string = g_strdup_printf ("line_count=%d", line_count);
    NOOP ("DF: %s\n", df_string);
    return df_string;
}

static void *
fstab_entry_tip (void *p) {
    record_entry_t *en = (record_entry_t *) p;
    if(!en) return NULL;
    gchar *module_text = NULL;
    
    gchar *mnt_dir = NULL;
    gchar *mnt_fsname = NULL;
    if(IS_PARTITION_TYPE (en->type)) {
        mnt_fsname = g_strdup (en->path);
        if(en->tag) {
            mnt_dir = g_strdup (en->tag);
        } else {
            mnt_dir = fstab_get_mnt_dir (mnt_fsname);
        }
    } else {
	/*if (!IS_LOCAL_TYPE(en->type)) {
	    return g_strdup (_("Remote Connection..."));
	}*/
        mnt_dir = g_strdup (en->path);
        if(en->tag) {
            mnt_fsname = g_strdup (en->tag);
        } else {
            mnt_fsname = fstab_get_mnt_fsname (mnt_dir);
        }
    }

    if(!mnt_dir)
        mnt_dir = g_strdup ("");
    if(!mnt_fsname)
        mnt_fsname = g_strdup ("");
    module_text =
        g_strdup_printf ("%s %s\n%s: %s\n%s %s: %s", 
                _("Mount point:"), mnt_dir, 
                _("Partition"), mnt_fsname,
                _("Partition type:"), 
		_("Mount"),
                (IS_USER_TYPE (en->type)) ? _("User") : _("Root"));
    g_free (mnt_dir);
    g_free (mnt_fsname);

    return module_text;
}


