#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
// Condition variable:
static gboolean glob_done = FALSE;
// Condition mutex:
static GMutex *glob_mutex = NULL;
// Condition signal:
static GCond *glob_signal = NULL;

// Hash tables:
static GHashTable *icon_hash = NULL;
static GHashTable *icon_exec_hash = NULL;
static GHashTable *string_hash = NULL;
static GHashTable *reverse_string_hash = NULL;
static GHashTable *category_hash = NULL;
static gboolean unload_module = FALSE;

// Hash table mutexes:

static GMutex *
get_icon_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}
static GMutex *
get_exec_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}static GMutex *
get_string_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}
static GMutex *
get_category_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}
static GMutex *
get_popup_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}






// Single linked lists:
static GSList *category_list = NULL;
//static GSList *path_list = NULL;


static gint desktop_serial=0;

typedef struct dotdesktop_t {
    gchar *category;
    gchar *string;
    gchar *icon;
    GSList *application_list; 
    gboolean popup;
} dotdesktop_t;

static dotdesktop_t dotdesktop_v[]={
    // These are categories which will appear in popup menu, 
    // in the specified order. Category translation is refined. 
    // Icon specification is hard coded.
    {"Rodent", NULL, "xffm/stock_directory", 
	NULL, FALSE},
    {"Utility", N_("Accessories"), "xffm/stock_directory/compositeSE/emblem_applications", 
	NULL, TRUE},
    {"Graphics", N_("Graphics applications"), "xffm/stock_directory/compositeSE/emblem_graphics", 
	NULL, TRUE},
    {"System", N_("System Tools"), "xffm/stock_directory/compositeSE/emblem_display", 
	NULL, TRUE},
    {"Network", N_("Internet and Network"), "xffm/stock_directory/compositeSE/emblem_www", 
	NULL, TRUE},
    {"Game", N_("Games and amusements"), "xffm/stock_directory/compositeSE/emblem_game", 
	NULL, TRUE},
    {"Office", N_("Office Applications"), "xffm/stock_directory/compositeSE/emblem_oo", 
	NULL, TRUE},
    {"Development", N_("Tools for software development"), "xffm/stock_directory/compositeSE/emblem_application",
       	NULL, TRUE},
    {"AudioVideo", N_("Applications related to audio and video"), "xffm/stock_directory/compositeSE/emblem_music",
       	NULL, TRUE},
    {"Settings", N_("Personal preferences"),"xffm/stock_directory/compositeSE/emblem_preferences",  
	NULL, TRUE},

    // These categories will not appear in popup menu but
    // translation may be refined and icon hard coded.
    {N_("Documentation"), NULL, "xffm/stock_directory/compositeSE/emblem_help", 
	NULL, FALSE},
    {N_("Accessibility"), NULL, NULL, 
	NULL, FALSE},
    {"Qt", NULL, "xffm/stock_directory/compositeSE/emblem_desktop",
       	NULL, FALSE},
//    {"GTK", NULL, "xffm/stock_directory/compositeSE/emblem_GTK", 
    {"GTK", NULL, "xffm/stock_directory/compositeSE/emblem_desktop", 
	NULL, FALSE},
    {"X-XFCE", NULL, "xffm/stock_directory/compositeSE/emblem_desktop", 
	NULL, FALSE},
    {"GNOME", NULL, "xffm/stock_directory/compositeSE/emblem_desktop", 
	NULL, FALSE},
    {"KDE", NULL, "xffm/stock_directory/compositeSE/emblem_desktop", 
	NULL, FALSE}, 
    {"Science", NULL, "xffm/stock_directory/compositeSE/emblem_science", 
	NULL, FALSE}, 
    {"Audio", NULL, "xffm/stock_directory/compositeSE/emblem_speaker", 
	NULL, FALSE}, 
    {"Literature", NULL, "xffm/stock_directory/compositeSE/emblem_bookmark", 
	NULL, FALSE}, 
    {"Spreadsheet", NULL, "xffm/stock_directory/compositeSE/emblem_spreadsheet", 
	NULL, FALSE}, 
    {"Math", NULL, "xffm/stock_directory/compositeSE/emblem_math", 
	NULL, FALSE}, 
    {"Printing", NULL, "xffm/stock_directory/compositeSE/emblem_print", 
	NULL, FALSE}, 
    {"TerminalEmulator", N_("Terminal Emulator"), "xffm/stock_directory/compositeSE/emblem_terminal", 
	NULL, FALSE}, 
    {"Security", NULL, "xffm/stock_directory/compositeSE/emblem_keyhole", 
	NULL, FALSE}, 
    {"RasterGraphics", N_("Raster Graphics"), "xffm/stock_directory/compositeSE/emblem_graphics", 
	NULL, FALSE}, 
    {"VectorGraphics", N_("Scalable Vector Graphics"), "xffm/stock_directory/compositeSE/emblem_graphics", 
	NULL, FALSE},
    {"Internet", NULL, "xffm/stock_directory/compositeSE/emblem_www", 
	NULL, FALSE}, 
    {"Emulator", NULL, "xffm/stock_directory/compositeSE/emblem_computer", 
	NULL, FALSE}, 
    // These categories will not appear in popup menu
    // and the translation will be refined.
    {"WordProcessor", N_("Word Processors"), NULL, 
	NULL, FALSE},
    {"DesktopSettings", N_("Desktop settings"), NULL, 
	NULL, FALSE},
    {"Player", N_("Media Player"), NULL, 
	NULL, FALSE}, 
    {"DiscBurning", N_("Video Disc Recorder"), NULL, 
	NULL, FALSE},
    {"PackageManager", N_("Package Manager"), NULL, 
	NULL, FALSE},
    {"HardwareSettings", N_("Hardware"), "xffm/stock_directory/compositeSE/emblem_chardevice", 
	NULL, FALSE},
    {"InstantMessaging", N_("Instant Messaging"), NULL, 
	NULL, FALSE},
    {"ContactManagement", N_("Contacts"), NULL, 
	NULL, FALSE},
    {"WebBrowser", N_("Web browser"), "xffm/stock_directory/compositeSE/emblem_www", 
	NULL, FALSE},
    {"LogicGame", N_("Logic Games"), NULL, 
	NULL, FALSE},
    {"2DGraphics", N_("2D Graphics"), "xffm/stock_directory/compositeSE/emblem_graphics", 
	NULL, FALSE},
    {"Recorder", N_("Sound Recorder"), NULL, 
	NULL, FALSE},
    {"BoardGame", N_("Board Games"), "xffm/stock_directory/compositeSE/emblem_games", 
	NULL, FALSE},
    {"AudioVideoEditing", N_("Audio Editor"), "xffm/stock_directory/compositeSE/emblem_video", 
	NULL, FALSE},
    {"BlocksGame", N_("Blocks Games"), NULL, 
	NULL, FALSE},
    {"CardGame", N_("Card Games"), NULL, 
	NULL, FALSE},
    {"Filesystem", N_("File System"), NULL, 
	NULL, FALSE},
    {"FileTools", N_("Batch tools for multiple files"), NULL, 
	NULL, FALSE},
    {"FileManager", N_("File Management"), NULL, 
	NULL, FALSE},
    {"FileTransfer", N_("File Transfer"), NULL, 
	NULL, FALSE},
    {"RemoteAccess", N_("Remote"), NULL, 
	NULL, FALSE},

    {NULL, NULL, "", NULL, FALSE}
};

// These are regular categories with acceptable translations
const gchar *translation_v[]={
    N_("Audio"),
    N_("Music"),
    N_("Application"),
    N_("Viewer"),
    N_("Email"),
    N_("Calendar"),
    N_("Archiving"),
    N_("Compression"),
    N_("Calculator"),
    N_("TextEditor"),
    N_("Clock"),
    N_("Core"),
    N_("Monitor"),
    N_("Spreadsheet"),
    N_("Science"),
    N_("Math"),
    N_("Video"),
    N_("Security"),
    N_("Photography"),
    N_("Scanning"),
    N_("Printing"),
    N_("P2P"),
    N_("Dictionary"),
    N_("Mixer"),
    N_("TrayIcon"),
    NULL
};

static
const gchar *
icon_by_path(const gchar *path);

static gchar *
get_hash_key (const gchar * pre_key) {
    GString *gs = g_string_new (pre_key);
    gchar *key;
    key = g_strdup_printf ("%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    return key;
}

static
gchar * 
get_desktop_string(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gchar *value;
	key_file=g_key_file_new ();

	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		NOOP("get_desktop_string(): %s (%s)",error->message,file);
		g_error_free(error);
		return NULL;
	}

	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
	    return NULL;
	}
	value=g_key_file_get_locale_string    (key_file,"Desktop Entry",key,NULL,&error);
	if (error){
		DBG("%s (%s)\n",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	if (strcmp(key, "Exec")==0){
	    gchar *test_exec=g_strdup(value);
	    if (strchr(test_exec, ' ')) *strchr(test_exec, ' ') = 0;
	    gchar *command=g_find_program_in_path(test_exec);
	    if (!command) {
		NOOP("get_desktop_string(): %s -> %s not found in path: %s\n", value, test_exec, file);
		g_free(value);
		value=NULL;
	    }
	    g_free(test_exec);
	    g_free(command);
	}

	return value;
}

static
int 
get_desktop_bool(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gboolean value=FALSE;
	key_file=g_key_file_new ();
	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		NOOP("get_desktop_bool(): %s (%s)",error->message,file);
		g_error_free(error);
		return FALSE;
	}
	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
	    return FALSE;
	}
	value=g_key_file_get_boolean(key_file,"Desktop Entry",key,&error);
	if (error){
		DBG("%s (%s)\n",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	return value;
}

static
gboolean
execute_dot_desktop(widgets_t *widgets_p, gchar *path, GList *target_list)
{
    gchar *exec=rfm_natural(PLUGIN_DIR, "dotdesktop", path, "item_exec");
    gchar *targets=NULL;



    if (target_list) {
	GList *tmp=target_list;
	for (; tmp && tmp->data; tmp=tmp->next){
	    gchar *esc_string=rfm_esc_string((gchar *)tmp->data);
	    gchar *p=g_strconcat((targets)?targets:"", " ", esc_string, NULL);
	    g_free(esc_string);
	    g_free(targets);
	    targets=p;
	}
    }    

    if (exec){
	gchar *command=g_strdup(exec);
	if (strchr(exec,' ')) *strchr(exec,' ')=0;

	// this is to chop off -f %F and %U stuff which is
	// not processed by double click (but should be DnD)
	if (strchr(command, '%')) {
	    if (targets!=NULL) {
		NOOP("targets=%s\n", targets);
		// replace the funky F or U or whatever for non nerdified format
		*(strchr(command, '%')+1)='s';
		// construct the full command line
		gchar *p=g_strdup_printf(command, targets);
		g_free(targets);
		g_free(command);
		command=p;
		NOOP("command=%s\n", command);
	    } else {
		if (strchr(command,' ')) *strchr(command,' ')=0;
	    }
	} else if (targets!=NULL) {
		// format specifier not found.
		// Assume default argv layout.
		// construct the full command line
		gchar *p=g_strdup_printf("%s %s", command, targets);
		g_free(targets);
		g_free(command);
		command=p;
	}


	gchar *fullexec=g_find_program_in_path(exec);
	if (fullexec && rfm_g_file_test(fullexec, G_FILE_TEST_IS_EXECUTABLE)){
	    rfm_show_text(widgets_p);
	    gboolean in_terminal=GPOINTER_TO_INT(
		    rfm_natural(PLUGIN_DIR, "dotdesktop", path, "exec_in_terminal"));
	    RFM_THREAD_RUN2ARGV (widgets_p, command, in_terminal);
	    g_free(command);
	    g_free(exec);
	    g_free(fullexec);
	    return TRUE;
	} 
	gchar *text=g_strdup_printf(_("File \"%s\" does not exist or is not executable."), exec);
	rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text,
		    NULL, NULL);
	g_free(command);
	g_free(exec);
	g_free(fullexec);
    }
    return FALSE;
}



gboolean
put_icon_in_hash(const gchar *path, const gchar *icon){
    if (!icon || !path) return FALSE;
    gchar *iconpath=NULL;
    if (!rfm_g_file_test(icon, G_FILE_TEST_EXISTS)){
	gchar *basename = (g_path_is_absolute(icon))? g_path_get_basename(icon) : g_strdup(icon);
	if (strchr(basename, '.')) *strrchr(basename, '.')=0;
	iconpath=ICON_get_filename_from_basename(basename);
	g_free(basename);
	if (!iconpath) {
	    NOOP(stderr, "put_icon_in_hash(): %s not found in icon path\n", icon);
	    return FALSE;
	} 
    } else {
	iconpath = g_strdup(icon);
    }

    gchar *hash_key = get_hash_key(path);
    GMutex *icon_hash_mutex = get_exec_hash_mutex();
    g_mutex_lock(icon_hash_mutex);
    g_hash_table_replace (icon_hash, hash_key, g_strdup(iconpath));
    g_mutex_unlock(icon_hash_mutex);	

    gchar *exec=get_desktop_string("Exec", path);
    if (exec) {
	if (strchr(exec, '%')) {
	    // replace the funky F or U or whatever for non nerdified format
	    *(strchr(exec, '%')+1)='s';
	}
	hash_key = get_hash_key(exec);
	GMutex *exec_hash_mutex = get_exec_hash_mutex();
	g_mutex_lock(exec_hash_mutex);
	g_hash_table_replace (icon_exec_hash, hash_key, g_strdup(iconpath));
	g_mutex_unlock(exec_hash_mutex);
	    NOOP(stderr, "hash insert %s: %s -> %s\n", path, exec, iconpath);
	
	// Hash the plain command as well, if applicable
	if (strchr(exec, ' ')){
	    *(strchr(exec, ' ')) = 0;
	    hash_key = get_hash_key(exec);
	    NOOP("hash insert %s: %s -> %s\n", path, exec, iconpath);
	    g_mutex_lock(exec_hash_mutex);
	    g_hash_table_replace (icon_exec_hash, hash_key, g_strdup(iconpath));
	    g_mutex_unlock(exec_hash_mutex);
	}
	g_free(exec);
    }
    g_free(iconpath);
  
    return TRUE;
}

static gpointer
glob_dir_f( gpointer data){

    glob_t stack_glob_v;
    gchar *prefix[]={NULL, "/usr", "/usr/local", NULL};
    prefix[0] = (gchar *)g_get_user_data_dir();

    gint i;
    // don't hog up the disk on startup.
    g_thread_yield();
    for (i=0; i<5; i++) rfm_threadwait();
    //forcing bug condition now for 10 seconds...
    //sleep(10);

    for (i=0; prefix[i]; i++){
	gint flags;
	gchar *directory=g_strdup_printf("%s/share/applications/*.desktop", prefix[i]);
	if (i==0) flags = 0;
	else flags =  GLOB_APPEND;
	//gint glob_result = 
	glob(directory, flags, NULL, &stack_glob_v);
	g_free(directory);
    }

    gint gi;
    GMutex *string_hash_mutex = get_string_hash_mutex();
    GMutex *category_hash_mutex = get_category_hash_mutex();
    for (gi=0; gi < stack_glob_v.gl_pathc; gi++){
	NOOP("glob=%s\n", stack_glob_v.gl_pathv[gi]);
	GKeyFile *key_file;
	GError *error=NULL;
	key_file=g_key_file_new ();

	const gchar *key="Categories";
	if (!g_key_file_load_from_file (key_file, stack_glob_v.gl_pathv[gi], 0, &error)){
		NOOP("%s (%s)",error->message,stack_glob_v.gl_pathv[gi]);
		g_error_free(error);
		continue;
	}
	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
		continue;
	}
	gchar *value=g_key_file_get_string (key_file,"Desktop Entry",key,&error);
	if (error){
		DBG("%s (%s)\n",error->message,stack_glob_v.gl_pathv[gi]);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	//gchar **values=rfm_split(value, ';');
	NOOP("value=%s\n", value);
	gchar **values=g_strsplit(value, ";", -1);

	gchar **x;
	gchar bugfixer='0'; // hack to zapout duplicate categories listed.
	for (x=values; x && *x; x++){
	    gchar **y=x+1;
	    for (;y && *y; y++){
		if (strcmp(*x, *y)==0){
		    // incorrect. Duplicate category...
		    TRACE("duplicate category: %s in %s (broken upstream desktop file)\n", *y, stack_glob_v.gl_pathv[gi]);
		    *(*y)=bugfixer;
		}
	    }
	}
	g_free(value);

// XXX We need a bug workaround if incorrect key file has repeated category.

// create array of standard categories, data is gslist
// foreach category add path to desktop file to category glist.
// categories will be the icons of the module,
// on each icon click, we will get the paths of all items in the categories for
//  the xfdir structure. No monitor enabled, or monitor globbed directories and reglob
//  on modifications....

	gchar **p=values;
	for (; p && *p; p++){
	    if (strlen(*p)==0) continue;
	    if (*(*p)==bugfixer) continue; // this takes care of ignoring duplicate categories
	    g_mutex_lock(category_hash_mutex);
	    dotdesktop_t *dotdesktop_p=g_hash_table_lookup(category_hash, *p);
	    g_mutex_unlock(category_hash_mutex);
	    if (dotdesktop_p == NULL) {
		// Add new category
		NOOP("new category=%s\n", *p);
		NOOP("    {N_(\"%s\"), N_(\"\")},\n", *p);
		gchar *cat=g_strdup(*p);
	    
		dotdesktop_p = (dotdesktop_t *) malloc(sizeof(dotdesktop_t));
		if (!dotdesktop_p) g_error("malloc: %s", strerror(errno));
		g_mutex_lock(category_hash_mutex);
		g_hash_table_replace(category_hash, g_strdup(cat), dotdesktop_p);
		g_mutex_unlock(category_hash_mutex);
		memset(dotdesktop_p, 0, sizeof(dotdesktop_t));
		dotdesktop_p->category=g_strdup(cat);

		dotdesktop_t *qq=dotdesktop_v;
		for (; qq && qq->category; qq++){
		    if (strcmp(qq->category, cat)==0) {
			dotdesktop_p->popup=qq->popup;
			dotdesktop_p->string=(qq->string)?g_strdup(qq->string):NULL;
			dotdesktop_p->icon=(qq->icon)?g_strdup(qq->icon):NULL;
			break;
		    }
		}
		// If no icon is assigned, then use the icon of the
		// first dot desktop element found in that category...
		if (dotdesktop_p->icon==NULL) {
		    dotdesktop_p->icon = g_strdup(icon_by_path(stack_glob_v.gl_pathv[gi]));
		}
	        g_mutex_lock(string_hash_mutex);
		category_list = g_slist_prepend(category_list, dotdesktop_p);
		if (dotdesktop_p->string) {
		    g_hash_table_replace(reverse_string_hash, g_strdup(_(dotdesktop_p->string)), g_strdup(cat));
		} else {
		    g_hash_table_replace(reverse_string_hash, g_strdup(_(cat)), g_strdup(cat));
		}
	        g_mutex_unlock(string_hash_mutex);
		g_free(cat);
	    }
	    // add item to category 
		
	    dotdesktop_p->application_list = 
			g_slist_prepend(dotdesktop_p->application_list,
				g_strdup(stack_glob_v.gl_pathv[gi]));

	}
	g_strfreev(values);


    }
    globfree(&stack_glob_v);

    
    g_mutex_lock(glob_mutex);
    glob_done = TRUE;
    g_cond_broadcast(glob_signal);
    g_mutex_unlock(glob_mutex);
    NOOP("dotdesktop init... glob_done\n");

    return NULL;
}


static
void *
full_init(void){
    NOOP(stderr, "fullinit...\n");
    g_mutex_lock(glob_mutex);
    if (!glob_done) g_cond_wait(glob_signal, glob_mutex);
    g_mutex_unlock(glob_mutex);
    NOOP(stderr, "fullinit OK\n");
    return GINT_TO_POINTER(1);
}

static 
void * 
private_get_xfdir(xfdir_t *xfdir_p){
    full_init();
    // 1.
    // glob /usr/share/applications and /usr/local/share/applications
    // for dotdesktop files
    // 2.
    // Create/append categories hash
    //   key:
    //      - category name
    //   data:
    //      - category name (translatable)
    //      - category gslist
    // 3.Create/append category gslist
    //   data:
    //      - name (locale dependent)
    //      - exec
    //      - comment (locale dependent)
    //      - create/add mimetype hash (this will feed applications module)
    //
    // 1.
    // glob /usr/share/applications and /usr/local/share/applications
    // for dotdesktop files

    //
    gint parent_category=0;
    
    NOOP(stderr, "dotdesktop load xfdir\n");
    record_entry_t *p_en=rfm_copy_entry(xfdir_p->en);
    if (p_en && p_en->st) {
    NOOP(stderr, "p_en && p_en->st 0x%x 0x%x\n", 
	    GPOINTER_TO_INT(p_en), GPOINTER_TO_INT(p_en->st));
	parent_category=p_en->st->st_uid;
	// this is very important, since xfdir_p->st gets
	// put into view_p->en->st, which is used for reload
	memcpy(xfdir_p->en->st, p_en->st, sizeof(struct stat));
    }


 
    if (parent_category==0) {
    NOOP(stderr, "dotdesktop No parent. These are categories\n");
	// No parent. These are categories.
	xfdir_p->pathc=1; 
	gint i;

	xfdir_p->pathc = g_slist_length(category_list) + 1;
	NOOP("pathc=%d\n", (gint)xfdir_p->pathc);


	xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
	if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
	memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
	// Up is rodent root level.
	xfdir_p->gl[0].en = NULL;
	NOOP("up set to root level\n");
	xfdir_p->gl[0].pathv = g_strdup (g_get_host_name ());

	GSList *tmp=category_list;
	gint id=1;
	GMutex *string_hash_mutex = get_string_hash_mutex();
	for (i=1;tmp && tmp->data; tmp=tmp->next,id++){
	    dotdesktop_t *category_p=tmp->data;
	    // Translation:
	    g_mutex_lock(string_hash_mutex);
	    const gchar *string=g_hash_table_lookup(string_hash, category_p->category);
	    g_mutex_unlock(string_hash_mutex);
	    if (!string) string = category_p->category;

	    xfdir_p->gl[i].pathv = g_strdup(_(string));
	    xfdir_p->gl[i].en=rfm_mk_entry(0);
	    xfdir_p->gl[i].en->type=0; /* remove local-type attributes */
	    xfdir_p->gl[i].en->parent_module = MODULE_NAME;
	    xfdir_p->gl[i].en->module = MODULE_NAME;
	    xfdir_p->gl[i].en->path = g_strdup(_(string));

	    xfdir_p->gl[i].en->st = (struct stat*) malloc(sizeof(struct stat));
	    if (!xfdir_p->gl[i].en->st) g_error("malloc: %s", strerror(errno));
	    memset(xfdir_p->gl[i].en->st, 0, sizeof(struct stat));
	    xfdir_p->gl[i].en->st->st_uid=id;
	    //xfdir_p->gl[i].en->st->st_mtime=1;//hack.
	    NOOP(stderr, "path=%s module=%s\n", 
			xfdir_p->gl[i].en->path, xfdir_p->gl[i].en->module);
	    i++;
	}
    } else {
	parent_category--;
	GSList *tmp = g_slist_nth(category_list, parent_category);
	GSList *list=NULL;
	if (tmp) {
	    dotdesktop_t *q=tmp->data;
	    NOOP("loading category %s\n", q->category); 
	    list=q->application_list;

	    xfdir_p->pathc = g_slist_length(list)+1;
	} else {
	    xfdir_p->pathc = 1;
	}
	NOOP("category=%d, pathc=%d\n", parent_category, (gint)xfdir_p->pathc);
	// up item is module root 
	// xfdir_p->pathc++;

	xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
	if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
	memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
	xfdir_p->gl[0].pathv = g_strdup(MODULE_LABEL);
	xfdir_p->gl[0].en=rfm_mk_entry(0);
	xfdir_p->gl[0].en->parent_module = MODULE_NAME;
	xfdir_p->gl[0].en->module = MODULE_NAME;
	xfdir_p->gl[0].en->st = NULL;	
	xfdir_p->gl[0].en->path=g_strdup(MODULE_LABEL);
        SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
	SET_UP_TYPE(xfdir_p->gl[0].en->type);

	gint i;
	for (i=1; list && list->data;list=list->next, i++){
	    gchar *path=list->data;
	    gchar *name=get_desktop_string("Name", path);
	    if (name) {
		xfdir_p->gl[i].pathv = name;
	    } else {
		xfdir_p->gl[i].pathv = g_path_get_basename(path);
	    }
	    xfdir_p->gl[i].en=rfm_stat_entry(path, 0);
	    xfdir_p->gl[i].en->type=0; 
	    xfdir_p->gl[i].en->module = MODULE_NAME;
	    xfdir_p->gl[i].en->path = g_strdup(path);
	    xfdir_p->gl[i].en->mimetype = g_strdup("application/x-desktop");

	    NOOP("category=%d path=%s\n",
		    parent_category, xfdir_p->gl[i].en->path);
	}

    }
    rfm_destroy_entry(p_en);
    return GINT_TO_POINTER(1);
}


static void
menu_exec(GtkMenuItem *m, gpointer data){
    gchar *path=data;
    widgets_t *widgets_p=rfm_get_widget("widgets_p");
    NOOP("execute %s", path);
    execute_dot_desktop(widgets_p, path, NULL);
}

	// Dotdesktop define icon for items.
static void *
populate_menuicons(gpointer data){
    GMutex *popup_mutex = get_popup_hash_mutex();
    g_mutex_lock(popup_mutex);
    GSList **menulist=data;
    GSList *tmp = *menulist;
    GMutex *icon_hash_mutex = get_icon_hash_mutex();
    for (;tmp && tmp->data; tmp=tmp->next){
	GtkWidget *menuitem = tmp->data;
	const gchar *path = g_object_get_data(G_OBJECT(menuitem), "path");
	gchar *hash_key = get_hash_key(path);
	g_mutex_lock(icon_hash_mutex);
	const gchar *iconpath = g_hash_table_lookup (icon_hash, hash_key);
	g_mutex_unlock(icon_hash_mutex);
	if (!iconpath) {
	    gchar *icon=get_desktop_string("Icon", path);
	    if (!icon) icon=g_strdup("xffm/emblem_exec");
	    put_icon_in_hash(path, icon);
	    g_free(icon);
	    g_mutex_lock(icon_hash_mutex);
	    iconpath = g_hash_table_lookup (icon_hash, hash_key);
	    g_mutex_unlock(icon_hash_mutex);

	}
	NOOP(stderr, "iconpath = %s\n", iconpath);
	g_free(hash_key);
	// Put pixbuf for icon into menu item.
	rodent_thread_mk_pixmap_menu((iconpath)?iconpath:"xffm/emblem_exec", menuitem, MENU_PIXMAP);
	//thread_mk_pixmap_menu((iconpath)?iconpath:"xffm/emblem_exec", menuitem, MENU_PIXMAP);
    }
    g_slist_free(*menulist);
    g_free(menulist);
    g_mutex_unlock(popup_mutex);
    return NULL;
}

static void *
populate_icon_hash_f(void *p){
     // wait for module initialization to complete...
    full_init();
    GSList *cat_list=category_list;
    for (; cat_list && cat_list->data; cat_list=cat_list->next){
	dotdesktop_t *dotdesktop_p=cat_list->data;
	GSList *app_list=dotdesktop_p->application_list;
	for (;app_list && app_list->data; app_list = app_list->next){
	    const gchar *path=app_list->data;
	    gchar *icon=get_desktop_string("Icon", path);
	    if (icon) {
		put_icon_in_hash(path, icon);
		g_free(icon);
	    }
	}
    }
	    
    return NULL;
}

static void *
populate_mimetype_hash_f(void *p){
     // wait for module initialization to complete...
    full_init();
    NOOP("populate_mimetype_hash_f()...\n");
    GSList *cat_list=category_list;
    for (; cat_list && cat_list->data; cat_list=cat_list->next){
	dotdesktop_t *dotdesktop_p=cat_list->data;
	GSList *app_list=dotdesktop_p->application_list;
	for (;app_list && app_list->data; app_list = app_list->next){
	    const gchar *path=app_list->data;
	    gchar *mimetypes=get_desktop_string("MimeType", path);
	    if (mimetypes) {
		gchar *command=get_desktop_string("Exec", path);
		if (command) {
		    if (strchr(command, '%')) {
			// replace the funky F or U or whatever for non nerdified format
			*(strchr(command, '%')+1)='s';
		    }
		    gchar **types = g_strsplit (mimetypes, ";", -1);
		    gchar **type_p=types;
		    for (;type_p && *type_p; type_p++){
			if (!strchr(*type_p, '/')) continue;
			NOOP("Appending mimecommand: %s --> %s\n", *type_p, command);
			rfm_rational(RFM_MODULE_DIR,"mime", (void *)(*type_p),
				(void *)command, "mime_append");
		    }		    
		    g_free(command);
		    g_strfreev(types);
		}
		
		g_free(mimetypes);
	    }
	}
    }
    // Now that we are done updating the mimetype applications,
    // we thread off a cache regeneration to get cache correct
    // on next start up.
	
    //rfm_void(RFM_MODULE_DIR,"mime", "mime_generate_cache");
    return NULL;
}

static void *
populate_submenu_f(void *data){
    void **arg = data;
    GtkWidget *parent = arg[0];
    gchar *name = arg[1];
    gchar *comment = arg[2];
    gchar *path = arg[3];
    GSList **menu_items_list = arg[4];
    gchar *iconpath = arg[5];
    GtkWidget *v = gtk_menu_item_new_with_label (name);
    if (comment){
	GdkPixbuf *pixbuf = rfm_get_pixbuf(iconpath, SIZE_DIALOG);
	rfm_add_custom_tooltip(v, pixbuf, comment);
    }
    gchar *object_path = g_strdup(path);
    g_object_set_data(G_OBJECT(v),"path",object_path);
    *menu_items_list = g_slist_prepend(*menu_items_list, v);

    // Connect signal and show.
    gtk_container_add (GTK_CONTAINER (parent), v);
    g_signal_connect ((gpointer) v, "activate", G_CALLBACK (menu_exec), object_path);
    gtk_widget_show (v);
    g_free(name);
    g_free(path);
    g_free(comment);
    return v;
}

// This is a thread function, must have GDK mutex set for gtk commands...
static void 
populate_submenu (GtkWidget *popup_widget){
     // wait for module initialization to complete...
	    TRACE("populate_submenu\n");
    full_init();
    GMutex *popup_mutex = get_popup_hash_mutex();
    g_mutex_lock(popup_mutex);

    GHashTable *populate_hash=g_hash_table_new(g_str_hash, g_str_equal);

    GtkWidget *w;
    GSList **menu_items_list =
	(GSList **)malloc(sizeof(GSList *));
    if (!menu_items_list) g_error("malloc: %s", strerror(errno));
    *menu_items_list=NULL;

    gint iv;
    GMutex *icon_hash_mutex = get_icon_hash_mutex();
    GMutex *string_hash_mutex = get_string_hash_mutex();
    for (iv=0; dotdesktop_v[iv].category != NULL; iv++){
	if (!dotdesktop_v[iv].popup) {
	    NOOP(stderr, "!dotdesktop_v[iv].popup\n");
	    continue;
	}
	// Find category pointer, category_p
	GSList *tmp=category_list;
	dotdesktop_t *category_p=NULL;
	for (;tmp && tmp->data; tmp=tmp->next){
	    dotdesktop_t *qqq = tmp->data;
	    if (strcmp(dotdesktop_v[iv].category, qqq->category)==0){
		category_p=tmp->data;
		break;
	    }
	}
	if (!category_p) {
	    NOOP(stderr, "Category %s not found!\n", dotdesktop_v[iv].category);
	    continue;
	}
	NOOP(stderr, "dotdesktop_popup loop: %s\n", category_p->category);
	gchar *hash_key = get_hash_key(category_p->icon);
	g_mutex_lock(icon_hash_mutex);
	const gchar *iconpath = g_hash_table_lookup (icon_hash, hash_key);
	g_mutex_unlock(icon_hash_mutex);
	g_free(hash_key);
	if (!iconpath) put_icon_in_hash(category_p->category, category_p->icon);
	
	g_mutex_lock(string_hash_mutex);
	const gchar *string=g_hash_table_lookup(string_hash, category_p->category);
	g_mutex_unlock(string_hash_mutex);
	if (!string) string = category_p->category;

	NOOP(stderr, "%s: %s %s\n", category_p->category, _(string), category_p->icon);
	w = rodent_thread_add_submenu (popup_widget, _(string), NULL, category_p->icon);

	GSList *list=category_p->application_list;
	for (; list && list->data; list=list->next){
	    if (get_desktop_bool("NoDisplay", (gchar *)list->data)) continue;
	    
	    // Item will appear only in first menu category listed in dotdesktop file.
	    if (g_hash_table_lookup(populate_hash, list->data)){
		continue;
	    } else {
		g_hash_table_insert(populate_hash, list->data, GINT_TO_POINTER(1));
	    }
	    // Dotdesktop exec command for menu item.
	    gchar *exec=get_desktop_string("Exec", (gchar *)list->data);
	    if (!exec) {
		DBG("dotdesktop: broken desktop file %s has NULL exec!\n", (gchar *)list->data);
		continue;
	    }
	    if (strchr(exec,' ')) *strchr(exec,' ')=0;

	    // Dotdesktop defined name for item.
	    gchar *name=get_desktop_string("Name", (gchar *)list->data);
	    // Try translation broken option
	    if (!name) name=get_desktop_string("_Name", (gchar *)list->data);
	    if (!name){
	       DBG("dotdesktop: broken desktop file %s has NULL name!\n", (gchar *)list->data);
	       name = g_strdup(exec);
	    }
	    if (strcmp(name,exec)){
		gchar *text=g_strdup_printf("%s (%s)", name, exec);
		g_free(name);
		name=text;
	    } 
	    g_free(exec);

	    gchar *comment=get_desktop_string("Comment", (gchar *)list->data);
	    // Try translation broken option
	    if (!comment) comment=get_desktop_string("_Comment", (gchar *)list->data);
	    // Create menu item with retrieved name.
		
	    gchar *hash_key = get_hash_key((gchar *)list->data);
	    g_mutex_lock(icon_hash_mutex);
	    gchar *iconpath = g_hash_table_lookup (icon_hash, hash_key);
	    g_mutex_unlock(icon_hash_mutex);
	    g_free(hash_key);

	    void *arg[6];
	    arg[0] = w; // parent
	    arg[1] = name;
	    arg[2] = comment;
	    arg[3] = g_strdup((gchar *)list->data); //path
	    arg[4] = menu_items_list;
	    arg[5] = iconpath;

	    rfm_context_function(populate_submenu_f, arg);
	}
    }
    // Clean up popup hash table 
    g_hash_table_destroy(populate_hash);
    // Update menuitems icons.
    rfm_view_thread_create(NULL, populate_menuicons, menu_items_list, "populate_menuicons");

    g_mutex_unlock(popup_mutex);
    return ;
	
}

static void *
dotdesktop_double_click(void *p, void *q){
    record_entry_t *en=q;
    if (!en) {
	return NULL;
    }
    if (!en->path || !rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)) {
	return NULL;
    }
    widgets_t *widgets_p=p;
    execute_dot_desktop(widgets_p, en->path, NULL);
    return GINT_TO_POINTER(1); 
}

static void *
dotdesktop_entry_tip(void *p){
    record_entry_t *en=p;
    if (!en || !en->path ) return NULL;


    gchar *name=get_desktop_string("Name",en->path);
    gchar *generic_name=get_desktop_string("GenericName",en->path);
    gchar *exec=get_desktop_string("Exec",en->path);
    gboolean in_term=get_desktop_bool("Terminal",en->path);

    gchar *text = g_strconcat(
	    name,
	    (generic_name)?"\n(":"",
	    (generic_name)?generic_name:"",
	    (generic_name)?")":"",
	    "\n","\n", _("Command to run when clicked:"), " ", exec,
	    "\n","\n", _("Terminal application"), ": ",
	    (in_term)?_("Yes"):_("No"),
	    NULL);
    gchar *tip = rfm_utf_string(text);
    g_free(name);
    g_free(generic_name);
    g_free(exec);
    g_free(text);

    return (tip);
}

static
const gchar *
icon_by_path(const gchar *path){
    gchar *hash_key = get_hash_key(path);
    GMutex *icon_hash_mutex = get_icon_hash_mutex();
    g_mutex_lock(icon_hash_mutex);
    const gchar *iconpath = g_hash_table_lookup (icon_hash, hash_key);
    g_mutex_unlock(icon_hash_mutex);
    if (iconpath) {
	NOOP("%s: hashed icon=%s\n", path, iconpath);
	g_free(hash_key);
	return (void *)iconpath;
    }
    gchar *icon=get_desktop_string("Icon", path);
    NOOP("%s: icon=%s\n", path, iconpath);
    
    put_icon_in_hash(path, icon);
    // Just to be sure it actually is in icon_hash.
    g_mutex_lock(icon_hash_mutex);
    iconpath = g_hash_table_lookup (icon_hash, hash_key);
    g_mutex_unlock(icon_hash_mutex);
    g_free(icon);
    g_free(hash_key);
    if (!iconpath) {
	return "application/x-desktop";
    }
    return iconpath;
}

static
void *dotdesktop_item_icon_id(void *p){
    if (p==NULL){
	return "xffm/emblem_exec";
    } 
    record_entry_t *en=(record_entry_t *)p;
    if (IS_UP_TYPE(en->type)) return "xffm/stock_go-up";
    NOOP("resolving icon for %s (%s)\n", en->path, en->mimetype);
    gboolean icon_module=GPOINTER_TO_INT(rfm_void(RFM_MODULE_DIR, "icons", "module_active"));
    if (en->mimetype && strcmp(en->mimetype,"application/x-desktop")==0) {
	if (!icon_module) return "xffm/stock_file/compositeNE/stock_execute";
	return (void *) icon_by_path(en->path);
    }
    // By this point we are dealing with a category.
    if (!icon_module) return "xffm/stock_directory/compositeC/stock_execute";


    // Input value p is the translated category string.
    GMutex *string_hash_mutex = get_string_hash_mutex();
    g_mutex_lock(string_hash_mutex);
    const gchar *string = g_hash_table_lookup(reverse_string_hash, en->path);
    g_mutex_unlock(string_hash_mutex);
    
    if (!string){
	NOOP("dotdesktop_item_icon_id(): string==NULL (%s)\n", en->path);
	string=en->path;
    }
    // Try hard coded categories first.
    dotdesktop_t *category_p=dotdesktop_v;
    GMutex *icon_hash_mutex = get_icon_hash_mutex();
    for (; category_p && category_p->category; category_p++){
	if (en && strcasecmp(string, category_p->category)==0){
	    if (!category_p->icon){
		break;
	    }

	    NOOP("got icon:%s\n",category_p->icon);
	    gchar *hash_key = get_hash_key(_(category_p->category));
	    g_mutex_lock(icon_hash_mutex);
	    const gchar *iconpath = g_hash_table_lookup (icon_hash, hash_key);
	    g_mutex_unlock(icon_hash_mutex);
	    g_free(hash_key);
	    if (!iconpath) put_icon_in_hash(_(category_p->category), category_p->icon);
	    g_mutex_lock(icon_hash_mutex);
	    iconpath = g_hash_table_lookup (icon_hash, hash_key);
	    g_mutex_unlock(icon_hash_mutex);

	    if (iconpath) {
		NOOP("returning icon: %s\n", iconpath);
		return (void *)iconpath;
	    }
	}
    }

#if 10
    // Soft coded categories: index is by translated string
    // category_hash is nontranslated, basic.

    GMutex *category_hash_mutex = get_category_hash_mutex();
    g_mutex_lock(category_hash_mutex);
    dotdesktop_t *dotdesktop_p=g_hash_table_lookup(category_hash, string);
    g_mutex_unlock(category_hash_mutex);
    if (dotdesktop_p && dotdesktop_p->icon){
	if (g_path_is_absolute(dotdesktop_p->icon)) return "xffm/stock_directory/compositeSE/emblem_mouse";
	return (void *)dotdesktop_p->icon;
    }
    NOOP("no icon for %s\n", string);
    if (en->st == NULL){
	// this is the module root item
	static gchar *icon=NULL;
	if (icon == NULL){
	    icon = g_strdup_printf("%s/pixmaps/rodent-dotdesktop.svg", PACKAGE_DATA_DIR);
	}
	return icon;
    }
    NOOP("dotdesktop_item_icon_id(): could not find icon:%s\n",en->path);
    return "xffm/emblem_exec";
#endif
}

static void *
dotdesktop_hide_local_menu_items (void *p, void *q) {
    //widgets_t *widgets_p = p;
    if(!p)
        return NULL;


    gchar *symbols[] = {
        "duplicate_menuitem",
        "symlink_menuitem",
        "touch_menuitem",
        "rename_menuitem",
        "paste_menuitem",
	"sort1",
	"select_menu",
	"paste_menuitem",
	"edit_separator",
	"view_separator",
        NULL
    };
    gchar **pp;
    for(pp = symbols; *pp; pp++) {
	NOOP("hiding: %s\n", *pp);
        HIDE_IT (*pp);
    }
    return GINT_TO_POINTER (1);
}

static off_t
basedir_sum (const gchar * dir) {
    off_t sum = 0;


    NOOP ("dotdesktop: recurse_basedir_sum(%s)\n", dir);
    if(rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
	if(!rfm_g_file_test (dir, G_FILE_TEST_IS_SYMLINK)) {
	    struct stat st;
	    if(stat (dir, &st) == 0) {
		sum += st.st_mtime;
		sum += st.st_dev;
	    }
	}
    }
    return sum;
}

static gpointer
monitor_f(gpointer data) {
    off_t o_status=0;
    sleep(5);
    gchar *prefix[]={PACKAGE_DATA_DIR, "/usr/share", "/usr/local/share", NULL};
    gchar *package_dir = g_build_filename(PACKAGE_DATA_DIR, "applications", NULL);
    GMutex *category_hash_mutex = get_category_hash_mutex();
    GMutex *popup_mutex = get_popup_hash_mutex();
    rfm_global_t *rfm_global_p = rfm_global();
    while (1){
	g_mutex_lock(rfm_global_p->status_mutex);
	gint status = rfm_global_p->status;
	g_mutex_unlock(rfm_global_p->status_mutex);
	if (status == STATUS_EXIT) return NULL;
	
	g_mutex_lock(glob_mutex);
	if (!glob_done) g_cond_wait(glob_signal, glob_mutex);
	g_mutex_unlock(glob_mutex);
	// Test for reload condition
	off_t new_status=0;
	gint i;
	for (i=0; prefix[i]; i++){
	    gchar *directory=g_build_filename(prefix[i], "applications", NULL);
	    if (i == 0 || strcmp(directory,package_dir) != 0) {
		new_status += basedir_sum(directory);
	    }
	    g_free(directory);
	}
	if (o_status == 0) {
	    o_status = new_status;
	}
	
	NOOP(stderr, "monitor_f(dotdesktop-module.i): o_status %ld = %ld\n",
		(glong) o_status, (glong)new_status);	
	if (o_status != new_status) {
	    o_status = new_status;
	    g_mutex_lock(popup_mutex);
	    desktop_serial++;
	    g_mutex_lock(glob_mutex);
	    glob_done=FALSE;
	    g_mutex_unlock(glob_mutex);
	    // Clear old values:
	    GSList *tmp1=category_list;
	    for (;tmp1 && tmp1->data; tmp1=tmp1->next){
		dotdesktop_t *q=tmp1->data;
		GSList *tmp=q->application_list;
		for (; tmp && tmp->data; tmp=tmp->next){
		    g_free(tmp->data);
		}
		g_slist_free(q->application_list);
		q->application_list=NULL;
		g_mutex_lock(category_hash_mutex);
		g_hash_table_steal(category_hash, q->category);
		g_mutex_unlock(category_hash_mutex);
		g_free(q);
	    }
	    g_slist_free(category_list);
	    category_list=NULL;
	    // Set new values:
	    glob_dir_f(NULL);

	    // Inform other threads that everthing is cool.
	    g_mutex_lock(glob_mutex);
	    glob_done=TRUE;
	    g_cond_broadcast(glob_signal);
	    g_mutex_unlock(glob_mutex);
	    g_mutex_unlock(popup_mutex);
	    NOOP("monitor_f(dotdesktop-module.i): Reloaded applications...\n");
	}
	g_mutex_lock(rfm_global_p->status_mutex);
	status = rfm_global_p->status;
	if (status != STATUS_EXIT){
	    gint load_timeout = 5;
	    if (!rfm_cond_timed_wait(rfm_global_p->status_signal, 
			rfm_global_p->status_mutex, load_timeout))
	    {
		NOOP(stderr, "Dot desktop monitor timeout loop\n");
	    } else {
		TRACE("Dot desktop monitor got the signal\n");
	    }
	}
	status = rfm_global_p->status;
	g_mutex_unlock(rfm_global_p->status_mutex);
	if (status == STATUS_EXIT) return NULL;
	if (unload_module) break;
    }
    g_free(package_dir);
    return NULL;
}


