/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "rodent.h"
#include "rfm_modules.h"
#include "fuse-group_options.h"
#include "fuse-common.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "ecryptfs"
//Not applicable:
//#define SUBMODULE_NAME "fuse"
#define MODULE_LABEL _("Ecryptfs (EFS)")
#define MODULE_ICON_ID "xffm/emblem_atom/compositeC/emblem_lock"
#define MODULE_ENTRY_TIP _("Encrypted filesystem")
#define MODULE_PLUGIN_INFO _("Ecrypt Volume")
#define PARENT_MODULE_NAME "ecrypt"
#define MODULE_PREFERENCES_KEY "RODENT-EFS"

#include "module-skeleton.h"

// Skeleton definitions

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY(MODULE_PREFERENCES_KEY)
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(FALSE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(MODULE_PLUGIN_INFO)
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
G_MODULE_EXPORT RFM_MODULE_RELOAD(TRUE)
G_MODULE_EXPORT RFM_MODULE_SKIPWAIT(TRUE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

// Superceded:
// G_MODULE_EXPORT RFM_ITEM_ICON_ID
// G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT
// g_module_check_init is now located at fuse-common.c

#define EFS_AUTHORIZATION _("Encryption Options")
#define EFS_TIP _("Encrypt Files")
#define ECRYPTFS_SIG _("Mount ecrypt signature")
#define ECRYPTFS_FNEK_SIG _("Filename ecrypt signature")
#define EFS_INFO1 _("Ecrypt Filesystem (EFS)")
#define EFS_INFO2 _("New EFS Link")
#define EFS_REMOTE_PATH _("Encrypted directory")
#define EPS_ENABLE_FILENAME_CRYPTO _("Encrypt filenames")
#define EPS_REQUIRES_SIGNATURE _("Requires ecryptfs signature")
#define EPS_PASSTHROUGH _("Plaintext passthrough")

// option dialogs:
    
#include "mount-options.i"
#include "ecryptfs-options.i"
#include "ecryptfs-dialog.i"

#if 0
G_MODULE_EXPORT
void *
module_icon_id(void){
    static gchar *icon=NULL;
    if (icon == NULL){
	icon = g_strdup_printf("%s/pixmaps/rodent-ecryptfs.svg", PACKAGE_DATA_DIR);
    }
    return icon;
}
#endif

// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
xfdir_t *
module_xfdir_get (void *p) {
    void *argv[] = {
	p,
	(void *)"mount.ecryptfs",
	(void *)"efs://",
	(void *)"FUSE_MOUNT_POINT",
	(void *)MODULE_NAME,
	(void *)EFS_AUTHORIZATION,
	NULL
    };
    return FUSE_xfdir(argv);
}

// this is a repeat...
G_MODULE_EXPORT
const gchar *
item_icon_id (void *p){
    void *argv[] = {
	p,
	(void *)MODULE_LABEL,
	(void *)module_icon_id(),
	(void *)EFS_AUTHORIZATION,
	NULL
    };
    return FUSE_icon(argv);
}

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void * p, void *q){
    record_entry_t *en = q;
    if (!en) return NULL;
    const gchar *url = en->pseudo_path;
    return FUSE_click((void *)confirm_efs_host_dialog, url, en, MODULE_NAME);
}
#if 10
static void
umount_host (GtkMenuItem * menuitem, gpointer user_data){
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en || !en->path) return;
    widgets_t *widgets_p = rfm_get_widget("widgets_p");;

    SET_MOUNTED_TYPE (en->type);
    FSTAB_fstab_mount (widgets_p, en);
}

// This is called by thread function. 
static void
establish_ecryptfs_option(widgets_t *widgets_p,
	const gchar *url, const gchar *line, 
	const gchar *option_string, const gchar *option_id)
{
    if (!url | !option_string || !option_id) return;
    if ( strstr (line, option_string) ){
      gchar *old_sig =
	  group_options_get_key_value (url, option_id);
      gchar *new_sig = g_strdup(
	      strstr (line, option_string)+strlen(option_string));
      if (strchr(new_sig, '\n')) *strchr(new_sig, '\n')=0;
      if (old_sig && strcmp(old_sig, new_sig)) {
	  rfm_threaded_diagnostics (widgets_p, "xffm/places_folder-locked", NULL);
	  rfm_threaded_diagnostics(widgets_p, "xffm_tag/red",g_strconcat( _("Invalid signature"), ": ", NULL));
	  rfm_threaded_diagnostics(widgets_p, "xffm_tag/green", g_strconcat(new_sig, NULL));
	  rfm_threaded_diagnostics(widgets_p, "xffm_tag/red", g_strconcat(" != ", NULL));
	  rfm_threaded_diagnostics(widgets_p, "xffm_tag/green",  g_strconcat(old_sig, "\n", NULL));
      } else if (!old_sig || strlen(old_sig)==0){
	  group_options_set_key_value (url, option_id, new_sig);
      }
      g_free(new_sig);
      g_free(old_sig);
    }
}

// This is a thread function, must have GDK mutex set for gtk commands...
static void 
stdout_f (void *user_data, void *stream, int childFD){
// thread function
    widgets_t *widgets_p = user_data;
    view_t *view_p = widgets_p->view_p;
    char *line;
    line = (char *)stream;
    NOOP ("FORK stdout: %s\n", line);

    if(line[0] == '\n')
        return;

    if(strncmp (line, "Tubo-id exit:", strlen ("Tubo-id exit:")) == 0) {
        gchar *string = rfm_diagnostics_exit_string(line);
        rfm_threaded_diagnostics(widgets_p, "xffm/stock_stop", string);
	    g_free(view_p->user_data);
	    view_p->user_data = NULL;
    
    } else if (strstr(line, "Select key type to use for newly created files:")){
          rfm_threaded_diagnostics (widgets_p,  "xffm/greyball", g_strdup(line));
          if (childFD > 0){
            gchar *r = "2\n";
            if (write(childFD, r, strlen(r)) < strlen(r)){
		DBG("ecryptfs: short write (%s)\n", strerror(errno));
	    }
          }
          rfm_threaded_diagnostics (widgets_p,  "xffm/greyball", g_strdup(" -- > 1\n"));

    } else {
//#ifdef DEBUG
          rfm_threaded_diagnostics (widgets_p, NULL, g_strdup(line));
//#endif
	  establish_ecryptfs_option(widgets_p, (gchar *)view_p->user_data, 
		  line, "ecryptfs_sig=", "ECRYPTFS_SIG");
	  establish_ecryptfs_option(widgets_p, (gchar *)view_p->user_data, 
		  line, "ecryptfs_fnek_sig=", "ECRYPTFS_FNEK_SIG");
    }
    return;
}



static void *
cleanup_passfile(void *data){
    if (!data) return NULL;
    gchar *passfile=data;
    struct stat st;
    if (stat (passfile, &st)<0) {
	DBG("Cannot stat password file %s to wipeout\n", passfile);
    } else {
	sleep(2);
	gint fd = open(passfile, O_RDWR);
	if (fd < 0){
	    DBG("Cannot open password file %s to wipeout\n", passfile);
	} else {
	    gint i;
	    // wipeout
	    for (i=0; i<st.st_size; i++){
		const gchar *null="";
		if (write(fd, null, 1));
	    }
	    close(fd);
	}
    }
    if (unlink(passfile)<0) {
	    DBG("Cannot unlink password file %s\n", passfile);
    }
    memset(passfile, 0, strlen(passfile));
    g_free(passfile);
    return NULL;
}
static gchar *
get_passfile(widgets_t *widgets_p, const gchar *passphrase_text){
    gchar *ptext = g_strdup_printf(_("Enter Passphrase for %s"), passphrase_text);
    gchar *passphrase = rfm_get_response (widgets_p, ptext, NULL, TRUE);
    g_free(ptext);

    gint fd = -1;
    gchar *passfile = NULL;
    if (passphrase && strlen(passphrase)){
	time_t seconds;
	time (&seconds);
	gint try=0;
retry:
	srand ((unsigned)seconds);
	gint divide = RAND_MAX / 10000;
	if (divide == 0) divide++;
	if((seconds = rand () / divide) > 100000L){
	    seconds = 50001;
	}
	passfile = g_strdup_printf("%s/.efs-%ld", g_get_home_dir(), (long)seconds);
	// if the file exists, retry with a different seudo-random number...
	if (rfm_g_file_test(passfile, G_FILE_TEST_EXISTS)){
	    if (seconds > 0) seconds--;
	    else seconds++;
	    if (try++ < 300) {
		g_free(passfile);
		goto retry;
	    } else {
		g_error("This is a what some people call \"a bean that weighs a pound\"\n");
	    }
	}
	NOOP(stderr, "passfile=%s on try %d\n", passfile, try);

	fd = open (passfile, O_CREAT|O_TRUNC|O_RDWR, 0600);
//	fd = open (passfile, O_CREAT|O_TRUNC|O_RDWR|O_SYNC|O_DIRECT, 0600);
	if (fd > 0) {
	    if (write(fd, (void *)"passwd=", strlen("passwd=")) < 0){
		DBG("write %s: %s\n", passfile, strerror(errno));
	    }
	    if (write(fd, (void *)passphrase, strlen(passphrase)) < 0){
		DBG("write %s: %s\n", passfile, strerror(errno));
	    }
	    memset(passphrase, 0, strlen(passphrase));
	    close(fd);
	} else {
	    DBG("cannot open %s: %s\n", passfile, strerror(errno));
	}

    }
    return passfile;
}

static
void
run_fork_finished_function (void *user_data) {
    widgets_t *widgets_p = user_data;
    cleanup_passfile(widgets_p->data);
}

static void
mount_error(widgets_t *widgets_p, const gchar *point){
    rfm_show_text(widgets_p);
    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error",NULL);
    rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr",
	g_strdup_printf(_("The file with pathname \"%s\" is not a directory."),
	    point));
}

//If you enabled filename encryption then pass an additional mount option: 
//ecryptfs_fnek_sig=XY, where XY is the same signature you provide with the 
//ecryptfs_sig option. 

void *
mount_url (widgets_t *widgets_p, const gchar *url){
	    TRACE( "efs: mount_url: %s\n", url);
    gchar **options_mount = NULL;
    gchar **options_efs = NULL;

    gchar *mount_point = group_options_get_key_value (url, "FUSE_MOUNT_POINT");
    if (!fuse_mkdir(mount_point)){
	mount_error(widgets_p, mount_point);
	g_free(mount_point);
	return NULL;
    }

    gchar *remote_path = group_options_get_key_value (url, "FUSE_REMOTE_PATH");

    if (!g_file_test(remote_path, G_FILE_TEST_IS_DIR)){
	mount_error(widgets_p, remote_path);
	g_free(mount_point);
	g_free(remote_path);
	return NULL;
    }
    gchar *argv[MAX_COMMAND_ARGS];
    const gchar *remote = url;
    if (strncmp(remote, "efs:///", strlen("efs:///"))==0) {
	remote += strlen("efs://");
    }
    
    gint i=0;

    if (geteuid() != 0) {
	argv[i++] = "sudo";
	argv[i++] = "-A";
    }
    
    argv[i++] = "mount";
    // Mount options
    options_mount = 
	group_options_get_key_options(url, 6, 
	    mount_options);
    gchar **o;
    gchar *m_options = NULL;
    for (o=options_mount; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (!m_options) m_options = g_strdup("-");
	gchar *p = *o;
	gchar *g=g_strconcat(m_options, p+1, NULL);
	m_options=g;
    }
    if (m_options) argv[i++] = m_options;

    argv[i++] = "-t";
    argv[i++] = "ecryptfs";


    // The rest of options...

    // efs options
    options_efs = 
	group_options_get_key_options(url, 12, 
	    efs_options);
    gchar *options=NULL;
    for (o=options_efs; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (strcmp(*o, "-o")==0) {
	    continue;
	}
	if (!options) options = g_strdup(*o);
	else {
	    gchar *g=g_strconcat(options, ",", *o, NULL);
	    g_free(options);
	    options=g;
	}
    }

    // default options, must be defined:
    if (!strstr(options, "ecryptfs_enable_filename_crypto=") ){
	    gchar *g=g_strconcat(options, ",ecryptfs_enable_filename_crypto=no", NULL);
	    g_free(options);
	    options=g;
    }
    if (!strstr(options, "ecryptfs_key_bytes=") ){
	    gchar *g=g_strconcat(options, ",ecryptfs_key_bytes=16", NULL);
	    g_free(options);
	    options=g;
    }
    if (!strstr(options, "ecryptfs_cipher=") ){
	    gchar *g=g_strconcat(options, ",ecryptfs_cipher=aes", NULL);
	    g_free(options);
	    options=g;
    }
    // end of default options.

    // signatures:
    //options = add_option(options, url, "ecryptfs_sig=", "ECRYPTFS_SIG");


    // if passphrase is not included, query for passphrase, put it in a temporary file
    // and send the file to mount command. Then erase temporary file after mount 
    // command executed.
    gchar *passfile = NULL;
    if (!strstr(options, "passphrase_") && !strstr(options, "openssl_") ){
	passfile = get_passfile(widgets_p, remote_path);
	if (passfile) {         
	    gchar *g = g_strdup_printf("%s,key=passphrase,passphrase_passwd_file=%s", options, passfile);
	    g_free(options);
	    options = g;
	} 
    } else if (strstr(options, "passphrase_") && !strstr(options, ",key=passphrase")){
	    gchar *g = g_strdup_printf("%s,key=passphrase", options);
	    g_free(options);
	    options = g;
        
    } else if (strstr(options, "openssl_") && !strstr(options, ",key=openssl")){
	    gchar *g = g_strdup_printf("%s,key=openssl", options);
	    g_free(options);
	    options = g;
    }



    if (options){
	argv[i++] = "-o";
	argv[i++] = options;
    }

    argv[i++] = remote_path;
    argv[i++] = mount_point;

    argv[i++] = NULL;

    rfm_show_text(widgets_p);
    
    gchar **p = argv;
    for (; p && *p; p++) NOOP( " %s", *p);
    NOOP( "\n");

    // copy url so stdout thread knows what group to access in keyfile.
    view_t *view_p = widgets_p->view_p;
    view_p->user_data = g_strdup(url);

    widgets_p->data = passfile; // Data for tubo_done_f() cleanup function

    rfm_thread_run_argv_full (
	  widgets_p, 
	  argv, 
	  FALSE, 
	  NULL, //&stdin_fd,  //NULL,
	  stdout_f,
	  NULL,
	  run_fork_finished_function // This function will cleanup 
	  );

    g_free(options);
    g_free(m_options);
    g_free(remote_path);
    g_free(mount_point);

    g_strfreev(options_mount);
    g_strfreev(options_efs);

    
    return GINT_TO_POINTER(1);
}

static void
mount_host (GtkMenuItem * menuitem, gpointer user_data){
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t *widgets_p = rfm_get_widget("widgets_p");
    mount_url(widgets_p, en->pseudo_path);
}
 
static void 
do_properties (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    const gchar *url = en->pseudo_path;
    FUSE_click((void *)confirm_efs_host_dialog, url, NULL, MODULE_NAME);

    return ;
}


// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Popup menu for modules should be set as view data on the paper object
// identified by "private_popup_menu".
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
     void *argv[] = {
	(void *)EFS_AUTHORIZATION,
	(void *)do_properties,
	(void *)mount_host,
	(void *)umount_host
    };
   return FUSE_popup(argv);
}



//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
    if (!en->path) return NULL;
    if (strcmp(en->path, EFS_AUTHORIZATION)==0){
	return g_strdup(EFS_TIP);
    }
    if (rfm_g_file_test(en->path, G_FILE_TEST_IS_DIR)){
	gchar *text =  g_strdup_printf("\n%s\n\n%s\n",
		en->path,
		_("The mount point used for the media device connection."));
	return text;
    }
    if (en->module  && strcmp(en->module, MODULE_NAME)){
	const gchar *text = rfm_void(PLUGIN_DIR, en->module, "module_entry_tip");
	return g_strdup(text);
    } 
    return g_strdup("fixme: efs-submodule.c");
}

#endif


